package com.jfrog.sysconf;

import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import static com.jfrog.sysconf.SysConfigConstants.ACCESS_SUFFIX;
import static com.jfrog.sysconf.SysConfigConstants.ARTIFACTORY_SUFFIX;
import static com.jfrog.sysconf.SysConfigConstants.BOOTSTRAP_KEY_READ_TIMEOUT_DEFAULT;
import static com.jfrog.sysconf.SysConfigConstants.DISTRIBUTION_SUFFIX;
import static com.jfrog.sysconf.SysConfigConstants.KEY_BOOTSTRAP_KEY_READ_TIMEOUT;
import static com.jfrog.sysconf.SysConfigConstants.KEY_JOIN_KEY;
import static com.jfrog.sysconf.SysConfigConstants.KEY_MASTER_KEY;
import static com.jfrog.sysconf.SysConfigConstants.KEY_MASTER_KEY_FILE;
import static com.jfrog.sysconf.SysConfigConstants.KEY_ROUTER_EXTERNAL_PORT;
import static com.jfrog.sysconf.SysConfigConstants.KEY_ROUTER_INTERNAL_PORT;
import static com.jfrog.sysconf.SysConfigConstants.METADATA_SUFFIX;
import static com.jfrog.sysconf.SysConfigConstants.ROUTER_DEFAULT_EXTERNAL_PORT;
import static com.jfrog.sysconf.SysConfigConstants.ROUTER_DEFAULT_INTERNAL_PORT;
import static com.jfrog.sysconf.SysConfigConstants.XRAY_SUFFIX;

/**
 * Helper methods to handle some common and more involved configurations.
 *
 * @author Yossi Shaul
 */
public class SysConfigHelper {
    private final SysConfig config;
    private final SysLayout sysLayout;

    SysConfigHelper(SysConfig config, SysLayout sysLayout) {
        this.config = config;
        this.sysLayout = sysLayout;
    }

    /**
     * Location of the master key file based on configured environment.
     * Note: file location doesn't mean that the file exist!
     *
     * @return The location of the master key file based on configured environment.
     */
    public String getMasterKeyFile() {
        return getConfiguredMasterKeyFile().orElseGet(this::getDefaultMasterKeyFile);
    }

    /**
     * Returns the value of the master key if exist. The value of the master key is searched in the normal searching
     * order with the addition of looking if it's also defined in another file defined by
     * {@link SysConfigHelper#getMasterKeyFile()}.
     *
     * @return The value of the master key if exist.
     */
    public Optional<String> getMasterKey() {
        return clearEmptyChars(config.get(KEY_MASTER_KEY).orElseGet(this::readMasterKeyFromFile));
    }

    /**
     * Returns the value of the join key if exist. The value of the join key is searched in the normal searching
     * order with the addition of looking if it's also defined in another file defined by
     * {@link SysConfigHelper#getJoinKeyFile()}.
     *
     * @return The value of the join key if exist.
     */
    public Optional<String> getJoinKey() {
        return clearEmptyChars(config.get(KEY_JOIN_KEY).orElseGet(this::readJoinKeyFromFile));
    }

    public Optional<JoinKeyData> getJoinKeyData() {
        return getJoinKeyDataFromSystemYaml()
                .or(this::getJoinKeyDataFromBootstrap)
                .or(this::getJoinKeyDataFromFile);
    }

    private Optional<JoinKeyData> getJoinKeyDataFromSystemYaml() {
        return config.get(KEY_JOIN_KEY)
                .flatMap(this::clearEmptyChars)
                .map(key -> JoinKeyData.builder().source("system resource").shouldVerify(true).key(key).build());
    }

    private Optional<? extends JoinKeyData> getJoinKeyDataFromFile() {
        return Optional.ofNullable(readJoinKeyFromFile())
                .flatMap(this::clearEmptyChars)
                .map(key -> JoinKeyData.builder().source(getJoinKeyFile()).shouldVerify(true).key(key).build());
    }

    private Optional<? extends JoinKeyData> getJoinKeyDataFromBootstrap() {
        String bootstrapJoinKeyFile = getBootstrapJoinKeyFile();
        return Optional.ofNullable(config.decrypt(KEY_JOIN_KEY, fileToString(new File(bootstrapJoinKeyFile))))
                .flatMap(this::clearEmptyChars)
                .map(key -> JoinKeyData.builder().source(bootstrapJoinKeyFile).shouldVerify(false).key(key)
                        .build());
    }

    /**
     * Location of the join key file based on configured environment.
     * Note: file location doesn't mean that the file exist!
     *
     * @return The location of the join key file based on configured environment.
     */
    public String getJoinKeyFile() {
        return getConfiguredJoinKeyFile().orElseGet(this::getDefaultJoinKeyFile);
    }

    public int getRouterExternalPort() {
        return config.getInt(KEY_ROUTER_EXTERNAL_PORT, ROUTER_DEFAULT_EXTERNAL_PORT);
    }

    /**
     * Build the router URL bases on configuration. Returns the default of not configured.
     *
     * @return the router URL. e.g., http://localhost:8046 (no trailing slash)
     */
    public String getRouterUrl() {
        int port = config.getInt(KEY_ROUTER_INTERNAL_PORT, ROUTER_DEFAULT_INTERNAL_PORT);
        return "http://localhost:" + port;
    }

    /**
     * @return the Artifactory URL going through the router. e.g., http://localhost:8046/artifactory (no trailing slash)
     * @see SysConfigHelper#getRouterUrl()
     */
    public String getArtifactoryUrl() {
        return getRouterUrl() + ARTIFACTORY_SUFFIX;
    }

    /**
     * @return the Xray URL when going through the router. e.g., http://localhost:8046/xray (no trailing slash)
     * @see SysConfigHelper#getRouterUrl()
     */
    public String getXrayUrl() {
        return getRouterUrl() + XRAY_SUFFIX;
    }

    /**
     * @return the Distribution URL when going through the router. e.g., http://localhost:8046/distribution
     * @see SysConfigHelper#getRouterUrl()
     */
    public String getDistributionUrl() {
        return getRouterUrl() + DISTRIBUTION_SUFFIX;
    }

    /**
     * @return the Access URL when going through the router. e.g., http://localhost:8046/access (no trailing slash)
     * @see SysConfigHelper#getRouterUrl()
     */
    public String getAccessUrl() {
        return getRouterUrl() + ACCESS_SUFFIX;
    }

    /**
     * @return the Metadata URL when going through the router. e.g., http://localhost:8046/metadata (no trailing slash)
     * @see SysConfigHelper#getRouterUrl()
     */
    public String getMetadataUrl() {
        return getRouterUrl() + METADATA_SUFFIX;
    }

    private Optional<String> getConfiguredMasterKeyFile() {
        return config.get(KEY_MASTER_KEY_FILE);
    }

    private String getDefaultMasterKeyFile() {
        return sysLayout.getProductEtcSecurity() + "/" + "master.key";
    }

    private Optional<String> getConfiguredJoinKeyFile() {
        return config.get(SysConfigConstants.KEY_JOIN_KEY_FILE);
    }

    public String getDefaultJoinKeyFile() {
        return sysLayout.getProductEtcSecurity() + "/" + "join.key";
    }

    public String getBootstrapJoinKeyFile() {
        return sysLayout.getProductBootstrapAccessEtcSecurity() + "/" + "join.key";
    }

    private String readJoinKeyFromFile() {
        return config.decrypt(KEY_JOIN_KEY, fileToString(new File(getJoinKeyFile())));
    }

    private String readMasterKeyFromFile() {
        return fileToString(new File(getMasterKeyFile()));
    }

    private Optional<String> clearEmptyChars(String value) {
        return Optional.ofNullable(StringUtils.trimToNull(value));
    }

    private String fileToString(File file) {
        if (file.exists()) {
            try {
                return Files.readString(file.toPath());
            } catch (IOException ioe) {
                throw new UncheckedIOException(ioe);
            }
        }
        return null;
    }

    public long getBootstrapKeyReadTimeoutMs() {
        return TimeUnit.SECONDS
                .toMillis(config.getLong(KEY_BOOTSTRAP_KEY_READ_TIMEOUT, BOOTSTRAP_KEY_READ_TIMEOUT_DEFAULT));
    }
}
