/*
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2019 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.jfrog.sysconf;

import org.jfrog.security.crypto.EncodingType;
import org.jfrog.security.crypto.EncryptionWrapper;
import org.jfrog.security.crypto.EncryptionWrapperFactory;
import org.jfrog.security.crypto.JFrogMasterKeyEncrypter;
import org.jfrog.security.crypto.encoder.EncryptedString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;

/**
 * Helper class to decrypt values from the sys config.
 *
 * @author Yossi Shaul
 */
public class SysConfigDecryptionHandler {
    private static final Logger log = LoggerFactory.getLogger(SysConfigDecryptionHandler.class);

    private final SysConfig config;
    private EncryptionWrapper legacyEncrypter;
    private JFrogMasterKeyEncrypter masterEncrypter;

    /**
     * Create new encryption handler to handle the loading of master key values decryption
     */
    SysConfigDecryptionHandler(SysConfig config) {
        this.config = config;
    }

    /**
     * @param key   Key of the value (for logging purposes)
     * @param value Potentially encrypted value
     * @return Decrypted value or same value if not encrypted
     */
    public String decrypt(String key, String value) {
        conditionallyInitEncrypters(key, value);
        if (isEncryptersConfigured()) {
            if (legacyEncrypter.isEncodedByMe(value)) {
                return legacyEncrypter.decryptIfNeeded(value).getDecryptedData();
            } else if (masterEncrypter.isEncryptedByMe(value)) {
                return masterEncrypter.decrypt(value);
            }
        }
        return value;
    }

    public boolean isEncrypted(String value) {
        return EncodingType.MASTER_LEVEL.isEncodedByMe(value) || EncryptedString.isEncodedByMe(value);
    }

    void createEncryptersIfMasterKeyExist() {
        Optional<String> masterKey = config.helper().getMasterKey();
        if (masterKey.isPresent()) {
            legacyEncrypter = EncryptionWrapperFactory.aesKeyWrapperFromString(masterKey.get());
            masterEncrypter = new JFrogMasterKeyEncrypter(masterKey.get());
        }
    }

    private boolean isEncryptersConfigured() {
        return legacyEncrypter != null && masterEncrypter != null;
    }

    private void conditionallyInitEncrypters(String key, String value) {
        if (!isEncryptersConfigured() && isEncrypted(value)) {
            createEncryptersIfMasterKeyExist();
            if (!isEncryptersConfigured()) {
                log.warn("Encrypted key found for key '{}' but master.key is not configured", key);
            }
        }
    }
}
