/*
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2019 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.jfrog.sysconf;

import com.google.common.collect.Maps;

import javax.annotation.Nonnull;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.jfrog.sysconf.SysConfigConstants.KEY_JF_PRODUCT_HOME;

/**
 * This class supports reading configuration from multiple sources with well defined priorities.
 * Priority is first given to service specific property and if not found this method will also check for the same
 * property in the shared sections. <p>
 * For instance, the search order for a property name "artifactory.url" will be:
 * <pre>
 * 1. artifactory.url equivalent in Java system properties
 * 2. artifactory.url equivalent in OS environment variables
 * 3. artifactory.url in the system.yaml file
 * 4. shared.url in equivalent Java system properties
 * 5. shared.url in equivalent OS environment variables
 * 6. shared.url in the system.yaml file
 * 7. artifactory.url in the defaults map if supplied
 * 8. shared.url in the defaults map if supplied
 * 9. default provided in the method call
 * </pre>
 *
 * @author Yossi Shaul
 */
public class SysConfig {
    public static final String SYS_PROP_JF_PRODUCT_HOME = SysPropsSource.toSysPropsKey(KEY_JF_PRODUCT_HOME);

    private final SysPropsSource sysPropsSource;
    private final EnvVarsSource envVarsSource;
    private final SysYamlSource yamlSource;
    private final Map<String, String> defaults;
    private final SysLayout sysLayout;
    private final SysConfigDecryptionHandler decryptionHandler;

    private SysConfig(String homeDirPath, InputStream sysYamlStream, Map<String, String> userDefaults,
            Map<String, String> testEnvVars) {
        sysPropsSource = new SysPropsSource();
        envVarsSource = testEnvVars == null ? new EnvVarsSource() : new EnvVarsSource(testEnvVars);
        sysLayout = new SysLayout(getOrDetectAbsoluteHomeDir(homeDirPath, sysPropsSource, envVarsSource), "system");
        defaults = defensiveCopyOfDefaults(userDefaults);
        yamlSource = createYamlSource(sysYamlStream);
        decryptionHandler = new SysConfigDecryptionHandler(this);
    }

    /**
     * Returns the value logically associated with this key.
     *
     * @param key The key to look for.
     * @return Optional value associated with this key.
     */
    public Optional<String> get(@Nonnull String key) {
        Objects.requireNonNull(key, "Config key cannot be null");
        // priorities 1,2,3
        String value = getFromNonDefaultSources(key);
        if (value == null) {
            // priorities 4,5,6
            value = getSharedFromNonDefaultSources(key);
        }
        // priority 7
        if (value == null) {
            value = defaults.get(key);
        }
        //priority 8
        if (value == null) {
            value = defaults.get(toSharedKey(key));
        }

        return Optional.ofNullable(decrypt(key, value));
    }

    /**
     * @param key The key to look for.
     * @return value associated with this key or the supplied default if value not found
     */
    public String get(String key, String defaultValue) {
        return get(key).orElse(defaultValue);
    }

    /**
     * @param key The key to look for.
     * @return value associated with this key as an int or the supplied default if value not found
     * @throws NumberFormatException if the value is not a valid int
     */
    public int getInt(String key, int defaultValue) {
        String result = get(key, String.valueOf(defaultValue));
        return Integer.parseInt(result);
    }

    /**
     * @param key The key to look for.
     * @return value associated with this key as a long or the supplied default if value not found
     * @throws NumberFormatException if the value is not a valid long
     */
    public long getLong(String key, long defaultValue) {
        String result = get(key, String.valueOf(defaultValue));
        return Long.parseLong(result);
    }

    /**
     * @param key The key to look for.
     * @return value associated with this key as an boolean or the supplied default if value not found. False is
     * returned if the value is not a valid {@link Boolean} value
     */
    public boolean getBoolean(String key, boolean defaultValue) {
        String result = get(key, String.valueOf(defaultValue));
        return Boolean.parseBoolean(result);
    }

    /**
     * @param key The key to look for.
     * @return value associated with this key. Throws an exception if key not found
     * @throws MissingKeyException if key not found
     */
    public String getOrFail(String key) {
        return get(key).orElseThrow(() -> new MissingKeyException(key));
    }

    public SysConfigHelper helper() {
        return new SysConfigHelper(this, sysLayout);
    }

    private String getFromNonDefaultSources(@Nonnull String key) {
        String value = sysPropsSource.get(key);
        if (value != null) {
            return value;
        }
        value = envVarsSource.get(key);
        if (value != null) {
            return value;
        }
        return yamlSource.get(key);
    }

    String toSharedKey(String key) {
        int firstPartIndex = key.indexOf('.');
        return firstPartIndex > 0 ? "shared" + key.substring(firstPartIndex) : key;
    }

    public Path getHomeDir() {
        return sysLayout.getHomeDir().toPath();
    }

    private String getSharedFromNonDefaultSources(@Nonnull String key) {
        String sharedKey = toSharedKey(key);
        return sharedKey.equals(key) ? null : getFromNonDefaultSources(sharedKey);
    }

    private Map<String, String> defensiveCopyOfDefaults(Map<String, String> defaults) {
        return defaults == null ? Maps.newHashMap() : Maps.newHashMap(defaults);
    }

    private SysYamlSource createYamlSource(InputStream sysYamlStream) {
        if (sysYamlStream == null) {
            sysYamlStream = getSystemYamlFromDefaultLocation();
        }
        return new SysYamlSource(sysYamlStream);
    }

    private InputStream getSystemYamlFromDefaultLocation() {
        FileInputStream sysYamlStream = null;
        File systemYamlFile = new File(sysLayout.getProductEtc(), "system.yaml");
        if (systemYamlFile.exists()) {
            try {
                sysYamlStream = new FileInputStream(systemYamlFile);
            } catch (FileNotFoundException e) {
                throw new UncheckedIOException(e);
            }
        }
        return sysYamlStream;
    }

    static Path detectJFrogProductHome(SysPropsSource propsSource, EnvVarsSource envSource) {
        // product home can be defined by user as sys prop or env var
        String productHome = propsSource.get(KEY_JF_PRODUCT_HOME);
        if (productHome == null) {
            productHome = envSource.get(KEY_JF_PRODUCT_HOME);
        }
        if (productHome == null) {
            throw new IllegalStateException(
                    "Failed to create system config. JFrog product home wasn't supplied and couldn't be detected in " +
                            "environment variable '" + EnvVarsSource.toEnvVarKey(KEY_JF_PRODUCT_HOME) + "' or in " +
                            "system property '" + SysPropsSource.toSysPropsKey(KEY_JF_PRODUCT_HOME) + "'");
        }
        return Paths.get(productHome);
    }

    /**
     * Resolve the home dir from system properties or environment variables
     *
     * @return home directory File
     */
    public static File detectAbsoluteHomeDir() {
        SysPropsSource sysPropsSource = new SysPropsSource();
        EnvVarsSource envVarsSource = new EnvVarsSource();
        return getOrDetectAbsoluteHomeDir(null, sysPropsSource, envVarsSource);
    }

    private static File getOrDetectAbsoluteHomeDir(String homeDir, SysPropsSource propsSource,
            EnvVarsSource envSource) {
        Path homeDirPath = homeDir == null ? detectJFrogProductHome(propsSource, envSource) : Paths.get(homeDir);
        return homeDirPath.toAbsolutePath().toFile();
    }

    String decrypt(String key, String value) {
        return decryptionHandler == null ? value : decryptionHandler.decrypt(key, value);
    }

    /**
     * Builder for the {@link SysConfig} class. The builder can build a new system config without programmatic
     * configuration assuming user defined the product home dir. From user defined dir the library can conclude the
     * location of various config files such as the <code>system.yaml</code>, <code>master.key</code> and
     * <code>join.key</code>.
     * User defaults can be provided as a fallback when specific value doesn't exist.
     * For more complex scenarios, including dev and testing, the builder allows programmatically setting the home
     * and the <code>system.yaml</code>.
     */
    public static class Builder {
        private String homeDir;
        private InputStream sysYamlStream;
        private Map<String, String> defaults;
        private Map<String, String> testEnVars;

        public SysConfig build() {
            return new SysConfig(homeDir, sysYamlStream, defaults, testEnVars);
        }

        /**
         * Build with the provided path as the JFrog product var path. If not provided, the library will detect the
         * home from system property <code>jf.product.home</code> or environment variable <code>JF_PRODUCT_HOME</code>
         */
        public Builder withHome(String jfrogProductHome) {
            homeDir = jfrogProductHome;
            return this;
        }

        /**
         * Build with the provided input stream of the system.yaml file. If not provided, the library will look for the
         * file at the default location <code>$JF_PRODUCT_HOME/etc/system.yaml</code>.
         */
        public Builder withSystemYaml(InputStream is) {
            sysYamlStream = is;
            return this;
        }

        /**
         * @see Builder#withSystemYaml(java.io.InputStream)
         */
        public Builder withSystemYaml(File systemYamlFile) {
            if (systemYamlFile.exists()) {
                try {
                    withSystemYaml(new FileInputStream(systemYamlFile));
                } catch (FileNotFoundException e) {
                    throw new UncheckedIOException(e);
                }
            }
            return this;
        }

        /**
         * Build with the optional map of default values. The default values are consulted according to the priorities
         * defined in {@link SysConfig}
         */
        public Builder withDefaults(Map<String, String> defaults) {
            this.defaults = defaults;
            return this;
        }

        /**
         * Build with the mock of environment variables for testing only!
         */
        public Builder withMockEnvVars(Map<String, String> envVars) {
            this.testEnVars = envVars;
            return this;
        }

    }
}
