/*
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2019 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.jfrog.sysconf;

import com.google.common.collect.Maps;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;

/**
 * Source for system configuration taken from the process environment variables.
 * This source is capitalizing all letters and prefixing all the keys with the JFrog expected prefix for environment  -
 * {@link EnvVarsSource#ENV_VAR_PREFIX}. Separators are modified from '.' to '_' and '-' are removed.<p>
 * For a key:
 * <pre>
 *      artifactory.db.url
 * </pre>
 * this source looks for an environment variable named:
 * <pre>
 *     JF_ARTIFACTORY_DB_URL
 * </pre>
 *
 * @author Yossi Shaul
 */
class EnvVarsSource {
    static final String ENV_VAR_PREFIX = "JF_";
    private final Map<String, String> jfrogEnvVars = Maps.newHashMap();

    EnvVarsSource() {
        this(System.getenv());
    }

    // constructor for tests only (hard to mock environment variables)
    EnvVarsSource(Map<String, String> envVars) {
        collectAllEnvVarsWithJfrogPrefix(envVars);
    }

    @Nullable
    String get(@Nonnull String key) {
        return jfrogEnvVars.get(toEnvVarKey(key));
    }

    static String toEnvVarKey(String key) {
        String envVarKey = key.replaceAll("\\.", "_");
        envVarKey = envVarKey.replaceAll("-", "");
        return ENV_VAR_PREFIX + envVarKey.toUpperCase();
    }

    private void collectAllEnvVarsWithJfrogPrefix(Map<String, String> envVars) {
        envVars.keySet().stream()
                .filter(k -> k.startsWith(ENV_VAR_PREFIX))
                .forEach(k -> jfrogEnvVars.put(k, envVars.get(k)));
    }
}
