/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.support.common.core.collectors;

import org.jfrog.support.common.config.ConfigFilesConfiguration;
import org.jfrog.support.common.config.SystemLogsConfiguration;
import org.jfrog.support.common.core.AbstractSpecificContentCollector;
import org.jfrog.support.common.core.exceptions.BundleConfigurationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

/**
 * Config files collector
 *
 * @author Michael Pasternak
 */
public abstract class ConfigFilesCollector extends AbstractSpecificContentCollector<ConfigFilesConfiguration> {

    private static final Logger log = LoggerFactory.getLogger(ConfigFilesCollector.class);

    public ConfigFilesCollector() {
        super("conf");
    }

    protected abstract List<File> getConfigFiles();
    protected abstract String[] getScrubStrings();

        /**
     * Collects SystemLogs
     *
     * @param configuration {@link SystemLogsConfiguration}
     * @param tmpDir        output directory for produced content
     * @return operation result
     */
    @Override
    protected boolean doCollect(ConfigFilesConfiguration configuration, File tmpDir) {
        List<File> configFiles = getConfigFiles();
        if (configFiles != null && configFiles.size() > 0) {
            try {
                configFiles.stream()
                        .filter(f -> Files.exists(f.toPath()))
                        .filter(f -> Files.isRegularFile(f.toPath()))
                        .forEach(f -> {
                            if (configuration.isHideUserDetails()) {
                                scrubUserDataToTempDir(f.toPath(), tmpDir);
                            } else {
                                copyToTempDir(f.toPath(), tmpDir);
                            }
                        });
                getLog().info("Collection of " + getContentName() + " was successfully accomplished");
                return true;
            } catch (Exception e) {
                return failure(e);
            }
        } else {
            getLog().debug("No items to work on");
            return false;
        }
    }

    @Override
    public Logger getLog() {
        return log;
    }

    /**
     * Makes sure configuration is valid
     *
     * @param configuration configuration to check
     * @throws BundleConfigurationException if configuration is invalid
     */
    @Override
    protected void doEnsureConfiguration(ConfigFilesConfiguration configuration)
            throws BundleConfigurationException {
        ;
    }

    private void scrubUserDataToTempDir(Path file, File tmpDir) {
        String[] stringToScrub = getScrubStrings();
        File outputFile = new File(tmpDir.getPath() + File.separator + file.getFileName());
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(outputFile))) {
            List<String> lines = Files.readAllLines(file);
            for (String line : lines) {
                boolean writeLine = true;
                for (String password : stringToScrub) {
                    if (line.contains(password)) {
                        writeLine = false;
                    }
                }
                if (writeLine) {
                    bw.write(line);
                    bw.newLine();
                }
            }
        } catch (IOException e) {
            log.error("Failed to scrub support bundle data, cause: {}", e.getMessage());
            log.debug("Failed to scrub support bundle data", e);
        }
    }
}
