/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.support.common.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.jfrog.support.common.CollectConfiguration;
import org.jfrog.support.common.ContentCollector;
import org.jfrog.support.common.core.exceptions.BundleConfigurationException;

import java.io.File;
import java.io.IOException;

/**
 * Generic ContentCollector
 *
 * @param <T> configuration type declaration
 *
 *
 * @author Michael Pasternak
 */
public abstract class AbstractContentCollector<T extends CollectConfiguration>
        implements ContentCollector<T> {

    private static final Logger log = LoggerFactory.getLogger(AbstractContentCollector.class);
    private final String contentName;

    /**
     * @param contentName a name for specific sub-folder
     */
    protected AbstractContentCollector(String contentName) {
        this.contentName = contentName;
    }

    /**
     * Collects content according to {@link CollectConfiguration}
     *
     * @param configuration {@link CollectConfiguration}
     * @param tmpDir output dir
     *
     * @return boolean
     */
    @Override
    public final boolean collect(T configuration, File tmpDir) {
        try {
            if (configuration.isEnabled()) {
                getLog().debug("Ensuring configuration is correct for '{}'", getContentName());
                doEnsureConfiguration(configuration);

                getLog().info("Starting " + getContentName() + " collection ...");
                File contentSpecificTmpDir = produceTempDirectory(tmpDir);
                ensureTempDir(contentSpecificTmpDir);
                return doCollect(configuration, contentSpecificTmpDir);
            } else {
                getLog().debug("Configuration {} is disabled", configuration);
            }
        } catch (IllegalStateException | IOException e) {
            getLog().error("Executing task {} " + getContentName() + " has failed ("+ e.getMessage()+")");
            getLog().debug("Cause: {}", e);
        }
        return false;
    }

    /**
     * Calculates temp directory from output directory and content-name
     *
     * @param outputDirectory the output directory
     *
     * @return temp directory
     */
    private File produceTempDirectory(File outputDirectory) {
        return new File(
                outputDirectory.getAbsolutePath() + File.separator + getContentName()
        );
    }

    /**
     * Makes sure directory exist
     *
     * @param archiveTmpDir
     */
    private void ensureTempDir(File archiveTmpDir) throws IOException {
        org.apache.commons.io.FileUtils.forceMkdir(archiveTmpDir);
    }

    /**
     * @return sub-folder name
     */
    protected String getContentName() {
        return contentName;
    }

    /**
     * Produces content specific output {@link File}
     *
     * @param tmpDir output dir
     * @return output {@link File}
     */
    protected File getOutputFile(File tmpDir) {
        return new File(tmpDir.getPath() + File.separator + getFileName());
    }

    /**
     * @return The filename to be used
     */
    protected String getFileName() {
        return getContentName()+".artifactory";
    }

    /**
     * @return logger
     */
    protected Logger getLog() {
        return log;
    }

    /**
     * Collects content according to {@link CollectConfiguration}
     *
     * @param configuration {@link CollectConfiguration}
     * @param tmpDir output dir
     *
     * @return boolean
     */
    protected abstract boolean doCollect(T configuration, File tmpDir);

    /**
     * Makes sure configuration is valid
     *
     * @param configuration configuration to check
     * @throws BundleConfigurationException if configuration is invalid
     */
    protected abstract void doEnsureConfiguration(T configuration) throws BundleConfigurationException;
}
