/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.common;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.codehaus.plexus.util.StringUtils;
import org.jfrog.common.ArgUtils;
import org.jfrog.common.ExecutionFailed;
import org.jfrog.common.RetryException;
import org.jfrog.common.Retryable;
import org.jfrog.common.TimeUnitFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.slf4j.event.Level;

@ThreadSafe
public abstract class ExecutionUtils {
    private static final Logger log = LoggerFactory.getLogger(ExecutionUtils.class);
    private static final Map<Level, Consumer<String>> logLevelTrace = Map.of(Level.DEBUG, arg_0 -> ((Logger)log).debug(arg_0), Level.INFO, arg_0 -> ((Logger)log).info(arg_0), Level.WARN, arg_0 -> ((Logger)log).warn(arg_0));

    private ExecutionUtils() {
    }

    public static <T> CompletableFuture<T> retry(Retryable<T> funcToRetry, RetryOptions retryOptions) {
        return ExecutionUtils.retry(funcToRetry, retryOptions, Executors.newCachedThreadPool());
    }

    public static <T> CompletableFuture<T> retry(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool) {
        CompletableFuture future = new CompletableFuture();
        Runnable task = ExecutionUtils.generateExecutionRunnable(funcToRetry, retryOptions, threadPool, future, 0, System.currentTimeMillis());
        threadPool.submit(task);
        return future;
    }

    private static <T> Runnable generateExecutionRunnable(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime) {
        return ExecutionUtils.addTraceId(() -> {
            try {
                ExecutionUtils.handleFunctionExecution(funcToRetry, retryOptions, threadPool, future, executedRetries, startTime);
            }
            catch (Throwable e) {
                ExecutionUtils.handleStopError(future, "code exception", retryOptions.getFailedLog(), e);
            }
        });
    }

    private static Runnable addTraceId(Runnable r) {
        String traceId = MDC.get((String)"uber-trace-id");
        if (StringUtils.isNotBlank((String)traceId)) {
            return new MDCRunnableDecorator(ExecutionUtils.getTraceIdFromHeaderStr(traceId), r);
        }
        return r;
    }

    public static String getTraceIdFromHeaderStr(String headerStr) {
        if (headerStr == null) {
            log.debug("Got null headerStr for traceId, returning \"\"");
            return "";
        }
        String traceId = headerStr;
        if (headerStr.contains(":")) {
            traceId = headerStr.split(":")[0];
        }
        return traceId;
    }

    private static <T> void handleFunctionExecution(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime) throws InterruptedException {
        try {
            future.complete(funcToRetry.tryExecute());
        }
        catch (RetryException e) {
            if (executedRetries + 1 >= retryOptions.numberOfRetries) {
                ExecutionUtils.handleStopError(future, "exceeded number of attempts (" + retryOptions.numberOfRetries + ")", retryOptions.getFailedLog(), e);
            }
            if (retryOptions.getShouldRetry().getAsBoolean()) {
                ExecutionUtils.handleRetry(funcToRetry, retryOptions, threadPool, future, executedRetries, startTime, e);
            }
            log.debug("Retry predicate failed, stop retrying");
            future.cancel(false);
        }
    }

    private static <T> void handleRetry(Retryable<T> funcToRetry, RetryOptions retryOptions, ExecutorService threadPool, CompletableFuture<T> future, int executedRetries, long startTime, RetryException e) throws InterruptedException {
        ExecutionUtils.log(retryOptions, executedRetries, e, startTime);
        Thread.sleep(retryOptions.timeout);
        int timeout = retryOptions.timeout * retryOptions.exponentialBackoffMultiplier;
        if (retryOptions.backoffMaxDelay > 0 && retryOptions.backoffMaxDelay < timeout) {
            timeout = retryOptions.backoffMaxDelay;
        }
        RetryOptions newOptions = RetryOptions.fromOther(retryOptions).timeout(timeout).build();
        threadPool.submit(ExecutionUtils.generateExecutionRunnable(funcToRetry, newOptions, threadPool, future, executedRetries + 1, startTime));
    }

    private static void log(RetryOptions retryOptions, int executedRetries, RetryException e, long startTime) {
        Level logLevel = ExecutionUtils.shouldLog(retryOptions, executedRetries, startTime) ? retryOptions.logLevel : Level.DEBUG;
        String elapsedTime = TimeUnitFormat.getTimeString(System.currentTimeMillis() - startTime, TimeUnit.MILLISECONDS);
        String formattedLog = StrSubstitutor.replace((Object)retryOptions.getRetryLog(), Map.of("RETRY", executedRetries + 1, "ELAPSED_TIME", elapsedTime, "ERROR", e.getMessage()));
        logLevelTrace.get(logLevel).accept(formattedLog);
    }

    private static boolean shouldLog(RetryOptions retryOptions, int executedRetries, long startTime) {
        return System.currentTimeMillis() > startTime + retryOptions.getReportDelayMillis() && (executedRetries + 1) % retryOptions.getBulkExceptionReportSize() == 0;
    }

    private static <T> void handleStopError(CompletableFuture<T> future, String reasonToFinish, String logTrace, Throwable exception) {
        String message = StrSubstitutor.replace((Object)logTrace, Map.of("ERROR", exception.getMessage(), "FINISH_REASON", reasonToFinish));
        log.debug(message, exception);
        log.error(message);
        future.completeExceptionally(new ExecutionFailed(message, exception));
    }

    public static class RetryOptions {
        private final int numberOfRetries;
        private final int exponentialBackoffMultiplier;
        private final int timeout;
        private final int backoffMaxDelay;
        private final int bulkExceptionReportSize;
        private final long reportDelayMillis;
        private final Level logLevel;
        private final String retryLog;
        private final String failedLog;
        private final BooleanSupplier shouldRetry;

        private RetryOptions(int numberOfRetries, int exponentialBackoffMultiplier, int timeout, int backoffMaxDelay, int bulkExceptionReportSize, long reportDelayMillis, Level logLevel, String retryLog, String failedLog, BooleanSupplier shouldRetry) {
            this.numberOfRetries = numberOfRetries;
            this.exponentialBackoffMultiplier = exponentialBackoffMultiplier;
            this.timeout = timeout;
            this.backoffMaxDelay = backoffMaxDelay;
            this.bulkExceptionReportSize = bulkExceptionReportSize;
            this.reportDelayMillis = reportDelayMillis;
            this.logLevel = logLevel;
            this.retryLog = retryLog;
            this.failedLog = failedLog;
            this.shouldRetry = shouldRetry;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static Builder fromOther(RetryOptions retryOptions) {
            return new Builder().numberOfRetries(retryOptions.numberOfRetries).exponentialBackoffMultiplier(retryOptions.exponentialBackoffMultiplier).timeout(retryOptions.timeout).backoffMaxDelay(retryOptions.backoffMaxDelay).bulkExceptionReportSize(retryOptions.bulkExceptionReportSize).reportDelayMillis(retryOptions.getReportDelayMillis()).logLevel(retryOptions.logLevel).retryLog(retryOptions.retryLog).failedLog(retryOptions.failedLog).shouldRetry(retryOptions.shouldRetry);
        }

        public int getNumberOfRetries() {
            return this.numberOfRetries;
        }

        public int getExponentialBackoffMultiplier() {
            return this.exponentialBackoffMultiplier;
        }

        public int getTimeout() {
            return this.timeout;
        }

        public int getBackoffMaxDelay() {
            return this.backoffMaxDelay;
        }

        public int getBulkExceptionReportSize() {
            return this.bulkExceptionReportSize;
        }

        public long getReportDelayMillis() {
            return this.reportDelayMillis;
        }

        public Level getLogLevel() {
            return this.logLevel;
        }

        public String getRetryLog() {
            return this.retryLog;
        }

        public String getFailedLog() {
            return this.failedLog;
        }

        public BooleanSupplier getShouldRetry() {
            return this.shouldRetry;
        }

        public static class Builder {
            private int numberOfRetries = 5;
            private int exponentialBackoffMultiplier = 2;
            private int timeout = 100;
            private int backoffMaxDelay = 1000;
            private int bulkExceptionReportSize = 1;
            private long reportDelayMillis;
            private Level logLevel = Level.WARN;
            private String retryLog = "Retry ${RETRY} Elapsed ${ELAPSED_TIME} failed: ${ERROR}. Trying again";
            private String failedLog = "Last retry failed: ${ERROR}. Not trying again (${FINISH_REASON})";
            private BooleanSupplier shouldRetry = () -> true;

            public Builder timeout(int timeout) {
                this.timeout = ArgUtils.requireNonNegative(timeout, "timeout must be non negative");
                return this;
            }

            public Builder numberOfRetries(int numberOfRetries) {
                this.numberOfRetries = ArgUtils.requirePositive(numberOfRetries, "numberOfRetries must be non negative");
                return this;
            }

            public Builder exponentialBackoffMultiplier(int exponentialBackoffMultiplier) {
                this.exponentialBackoffMultiplier = ArgUtils.requirePositive(exponentialBackoffMultiplier, "exponentialBackoffMultiplier must be positive");
                return this;
            }

            public Builder backoffMaxDelay(int backoffMaxDelay) {
                this.backoffMaxDelay = ArgUtils.requireNonNegative(backoffMaxDelay, "backoffMaxDelay must be non negative");
                return this;
            }

            public Builder bulkExceptionReportSize(int bulkExceptionReportSize) {
                this.bulkExceptionReportSize = ArgUtils.requireNonNegative(bulkExceptionReportSize, "backoffMaxDelay must be non negative");
                return this;
            }

            public Builder reportDelayMillis(long reportDelayMillis) {
                this.reportDelayMillis = ArgUtils.requireNonNegative(reportDelayMillis, "reportDelayMillis must be non negative");
                return this;
            }

            public Builder logLevel(Level logLevel) {
                this.logLevel = logLevel;
                return this;
            }

            public Builder retryLog(String retryLog) {
                this.retryLog = retryLog;
                return this;
            }

            public Builder failedLog(String failedLog) {
                this.failedLog = failedLog;
                return this;
            }

            public Builder shouldRetry(BooleanSupplier shouldRetry) {
                this.shouldRetry = shouldRetry;
                return this;
            }

            public RetryOptions build() {
                return new RetryOptions(this.numberOfRetries, this.exponentialBackoffMultiplier, this.timeout, this.backoffMaxDelay, this.bulkExceptionReportSize, this.reportDelayMillis, this.logLevel, this.retryLog, this.failedLog, this.shouldRetry);
            }
        }
    }

    static class MDCRunnableDecorator
    implements Runnable {
        private String traceId;
        private Runnable runnable;

        public MDCRunnableDecorator(String traceId, Runnable runnable) {
            this.traceId = traceId;
            this.runnable = runnable;
        }

        @Override
        public void run() {
            MDC.put((String)"uber-trace-id", (String)this.traceId);
            log.trace("starting a thread with traceId: {}", (Object)this.traceId);
            this.runnable.run();
        }
    }
}

