package org.jfrog.common.concurrency;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.TimeUnit;

/**
 * A util for application locking. To be used in conjunction with a {@link ConflictsGuard}
 *
 * @author Uriah Levy
 */
public class LockingUtilImpl implements LockingUtil {
    private static final Logger log = LoggerFactory.getLogger(LockingUtilImpl.class);

    @Override
    public <T> void clusterExclusive(ConflictsGuard<T> conflictsGuard, T lockKey, Runnable task) {
        ConflictGuard lock = conflictsGuard.getLock(lockKey);
        boolean lockAcquired = false;
        try {
            lockAcquired = lock.tryToLock(0, TimeUnit.SECONDS);
            if (lockAcquired) {
                log.debug("Lock acquired on key {}", lockKey);
                task.run();
                return;
            }
            log.debug("key '{}' is already locked.", lockKey);
        } catch (InterruptedException e) {
            log.error("Interrupted while waiting for lock on key '{}'.", lockKey, e);
            Thread.currentThread().interrupt();
        } finally {
            releaseLock(lockKey, lock, lockAcquired);
        }
    }

    @Override
    public <T> void unreleasableClusterExclusive(ConflictsGuard<T> conflictsGuard, T lockKey, Runnable task) {
        ConflictGuard lock = conflictsGuard.getLock(lockKey);
        boolean lockAcquired;
        try {
            lockAcquired = lock.tryToLock(0, TimeUnit.SECONDS);
            if (lockAcquired) {
                log.debug("Lock acquired on key {}", lockKey);
                task.run();
                return;
            }
            log.debug("key '{}' is already locked.", lockKey);
        } catch (InterruptedException e) {
            log.error("Interrupted while waiting for lock on key '{}'.", lockKey, e);
            lock.unlock();
            Thread.currentThread().interrupt();
        }
    }

    private <T> void releaseLock(T lockKey, ConflictGuard lock, boolean lockAcquired) {
        try {
            if (lockAcquired) {
                lock.unlock();
                log.debug("Lock on key '{}' released in finally block", lockKey);
            }
        } catch (Exception e) {
            log.error("Unable to release lock with key {}", lockKey, e);
        }
    }
}
