package org.jfrog.common;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.function.BiFunction;
import java.util.function.BinaryOperator;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.IntStream;
import java.util.stream.LongStream;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * Support takeWhile / Generate stream.
 *
 * @author saffih
 */
@SuppressWarnings("WeakerAccess")
public abstract class StreamSupportUtils {
    private static final Logger log = LoggerFactory.getLogger(StreamSupportUtils.class);

    private StreamSupportUtils() {
    }

    /**
     * RTFACT-15635 - when closing stream(SavedToFileOnReadInputStream type) which not fully read,
     * it will not move the file from _pre folder to the cache folder.
     * see SavedToFileInputStream#afterClose()
     *
     * @param stream - SavedToFileOnReadInputStream stream
     */
    public static void verifyStreamIsFullyRead(InputStream stream) {
        byte[] buffer = new byte[65536];
        try {
            int count = stream.read(buffer);
            while (count >= 0) {
                count = stream.read(buffer);
            }
        } catch (IOException e) {
            log.warn("Error occurred while trying to validate stream is fully read, reason {}", e.getMessage());
        }
    }

    /**
     * Enumeration to stream
     */
    public static <T> Stream<T> enumerationToStream(Enumeration<T> enumeration) {
        if ((enumeration == null) || !enumeration.hasMoreElements()) {
            return Stream.empty();
        }
        Stream<Enumeration<T>> generate = Stream.generate(() -> enumeration); // repeat
        Stream<Enumeration<T>> limited = StreamSupportUtils.takeWhile(generate, Enumeration::hasMoreElements); // limit
        return limited.map(Enumeration::nextElement);
    }

    /**
     * Autoclose - when the stream is fully consumed trigger stream.close()
     * .onClose on the internal stream!
     */
    public static <T> Stream<T> autoClose(Stream<T> aStream) {
        Stream<T> onConsume = Stream.of((T) null).peek(n -> aStream.close()).filter(ignore -> false);
        return Stream.of(aStream, onConsume).flatMap(i -> i);
    }

    /**
     * Generate stream, from generator method till it returns null.
     *
     * @param generator Function&lt;Integer,T&gt;
     * @return Stream&lt;T&gt;
     */
    public static <T> Stream<T> generateTillNull(Function<Integer, T> generator) {
        return StreamSupportUtils.takeWhile(Stream.iterate(0, i -> i + 1).map(generator));
    }

    public static <T> Stream<T> takeWhile(Stream<T> stream) {
        return takeWhile(stream, Objects::nonNull);
    }

    public static <T> Stream<T> takeWhile(Stream<T> stream, Predicate<? super T> predicate) {
        return StreamSupport.stream(takeWhile(stream.spliterator(), predicate), false);
    }

    // https://stackoverflow.com/questions/20746429/limit-a-stream-by-a-predicate
    //***  till we use java 1.9 ***/
    public static <T> Spliterator<T> takeWhile(
            Spliterator<T> splitr, Predicate<? super T> predicate) {
        return new Spliterators.AbstractSpliterator<T>(splitr.estimateSize(), 0) {
            boolean stillGoing = true;

            @Override
            public boolean tryAdvance(Consumer<? super T> consumer) {
                if (stillGoing) {
                    boolean hadNext = splitr.tryAdvance(elem -> {
                        if (predicate.test(elem)) {
                            consumer.accept(elem);
                        } else {
                            stillGoing = false;
                        }
                    });
                    return hadNext && stillGoing;
                }
                return false;
            }
        };
    }

    public static <T> Optional<T> retryUntil(int maxRetries, Supplier<T> producer, Predicate<T> passedCond) {
        return IntStream.range(0, maxRetries).boxed().map(i -> producer.get()).filter(passedCond).limit(1).findFirst();
    }

    // Idea of https://gist.github.com/kencharos/9884261
    public static <T> Stream<Stream<T>> chunk(Stream<T> source, int chunkSize) {
        final ArrayList<T> acc = new ArrayList<>(chunkSize);
        Function<T, Stream<T>> chunkAcc = (T it) -> {
            acc.add(it);
            if (acc.size() >= chunkSize) {
                try {
                    return new ArrayList<>(acc).stream();
                } finally {
                    acc.clear();
                }
            }
            return null;
        };

        Stream<Stream<T>> streamStream = source.map(chunkAcc).filter(Objects::nonNull);
        Stream<Stream<T>> last = Stream.of(acc).filter(it -> !it.isEmpty()).map(Collection::stream);

        return Stream.of(streamStream, last).flatMap(x -> x);
    }

    public static <T> Stream<List<T>> splitToLists(Stream<T> source, int chunkSize) {
        List<T> acc = new ArrayList<>(chunkSize);
        Function<T, List<T>> chunkAcc = (T it) -> {
            acc.add(it);
            if (acc.size() >= chunkSize) {
                try {
                    return new ArrayList<>(acc);
                } finally {
                    acc.clear();
                }
            }
            return null;
        };

        Stream<List<T>> streamStream = source.map(chunkAcc).filter(Objects::nonNull);
        Stream<List<T>> last = Stream.of(acc);

        return Stream.of(streamStream, last).flatMap(x -> x).filter(it -> !it.isEmpty());
    }

    public static <T> Stream<Pair<T, Long>> index(@Nonnull Stream<T> source) {
        return zip(source, LongStream.iterate(0, n -> n + 1).boxed());
    }


    public static <T, U> Stream<Pair<T, U>> zip(@Nonnull Stream<T> aStream, @Nonnull Stream<U> bStream) {
        return zip(aStream, bStream, Pair::of);
    }

    private static <T, U, R> Stream<R> zip(@Nonnull Stream<T> aStream, @Nonnull Stream<U> bStream,
            BiFunction<T, U, R> f) {
        Iterator<U> bIterator = bStream.iterator();
        return aStream.filter(x -> bIterator.hasNext()).map(x -> f.apply(x, bIterator.next()));
    }

    @SafeVarargs
    public static <T> Stream<T> merge(Comparator<T> ahead, Stream<T>... streams) {
        return mergeStreams(ahead, streams);
    }

    private static <T> Stream<T> mergeStreams(Comparator<T> ahead, @Nonnull Stream<T>[] streams) {
        BinaryOperator<Stream<T>> accumulator = (a, b) -> merge(a, b, ahead);
        return Arrays.stream(streams)
                .reduce(accumulator).orElse(Stream.empty());
    }

    public static <T> Stream<T> merge(Stream<T> a, Stream<T> b, Comparator<T> ahead) {
        MergeStreamsSpliterator<T> split = new MergeStreamsSpliterator<>(a, b, ahead);
        return StreamSupport.stream(split, false);
    }

    public static <T> Stream<T> stream(Collection<T> set) {
        return Optional.ofNullable(set)
                .orElse(Collections.emptySet())
                .stream();
    }

    public static <K, V> Stream<Map.Entry<K, V>> mapEntriesStream(Map<K, V> map) {
        map = Optional.ofNullable(map)
                .orElse(Collections.emptyMap());
        return stream(map.entrySet());
    }

    public static <K, V> Stream<Map.Entry<K, V>> multimapEntriesStream(Multimap<K, V> map) {
        map = Optional.ofNullable(map)
                .orElse(ArrayListMultimap.create());
        return stream(map.entries());
    }

    private static class MergeStreamsSpliterator<T> extends Spliterators.AbstractSpliterator<T> {
        private final Iterator<T> aIt;
        private final Iterator<T> bIt;
        private final Comparator<T> ahead;
        boolean hasA = false;
        boolean hasB = false;
        T storeA;
        T storeB;

        MergeStreamsSpliterator(Stream<T> a, Stream<T> b, Comparator<T> ahead) {
            super(Long.MAX_VALUE, 0);
            this.aIt = a.iterator();
            this.bIt = b.iterator();
            this.ahead = ahead;
        }

        @Override
        public boolean tryAdvance(Consumer<? super T> action) {
            // read missing to store
            if (!hasA && aIt.hasNext()) {
                hasA = true;
                storeA = aIt.next();
            }

            if (!hasB && bIt.hasNext()) {
                hasB = true;
                storeB = bIt.next();
            }
            // nothing to process
            if (!hasA && !hasB) {
                return false;
            }
            // accept head value by ahead function
            if (hasA && hasB) {
                int compare = ahead.compare(storeA, storeB);
                if (compare <= 0) {
                    hasA = false;
                    action.accept(storeA);
                } else {
                    hasB = false;
                    action.accept(storeB);
                }
            } else if (hasA) {
                hasA = false;
                action.accept(storeA);
            } else {// hasB
                hasB = false;
                action.accept(storeB);
            }

            return true;
        }

        @Override
        public Spliterator<T> trySplit() {
            return null;
        }

        @Override
        public long estimateSize() {
            return 0;
        }

        @Override
        public int characteristics() {
            return 0;
        }
    }
}
