package org.jfrog.common;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Stream;

/**
 * @author saffih
 */
public class ResultSetStream {
    private static final Logger log = LoggerFactory.getLogger(ResultSetStream.class);
    private ResultSet resultset;
    private int index = 0;


    public static <R> Stream<R> asStream(ResultSet resultset, ResultSetToObject<R> convert,
            CompletableFuture<SQLException> onException) {
        return new ResultSetStream(resultset, onException).asStreamObject(convert);
    }

    public static Stream<ResultSet> asStream(ResultSet resultset, CompletableFuture<SQLException> onException) {
        return new ResultSetStream(resultset, onException).asStream();
    }

    private CompletableFuture<SQLException> onException;

    private ResultSetStream(ResultSet resultset, CompletableFuture<SQLException> onException) {
        this.resultset = resultset;
        this.onException = onException;
    }

    public int getIndex() {
        return index;
    }

    @FunctionalInterface
    public interface ResultSetToObject<R> {
        R apply(ResultSet rs) throws SQLException;
    }

    private <R> Stream<R> asStreamObject(ResultSetToObject<R> convert) {
        return asStream().map(it -> {
            try {
                return convert.apply(it);
            } catch (SQLException e) {
                onException.complete(e);
                return null;
            }
        }).filter(Objects::nonNull);
    }

    public Stream<ResultSet> asStream() {
        if (resultset == null) {
            return Stream.empty();
        }

        Function<Integer, ResultSet> generator = i -> {
            try {
                if (resultset.next()) {
                    index = i;
                    return resultset;  // streaming
                }
            } catch (SQLException e) {
                if (onException != null) {
                    onException.complete(e);
                }
            } catch (Exception e) {
                close();
                throw e;
            }
            return null;
        };

        // stream.close closes the provider
        Stream<ResultSet> generated =
                StreamSupportUtils.generateTillNull(generator).filter(Objects::nonNull).onClose(this::close);
        return StreamSupportUtils.autoClose(generated);
    }

    private void close() {
        if (resultset != null) {
            try {
                resultset.close();
                resultset = null;
            } catch (SQLException e) {
                log.trace("Could not close JDBC result set", e);
            } catch (Exception e) {
                log.trace("Unexpected exception when closing JDBC result set", e);
            }
        }
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }
}