package org.jfrog.common;


import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.lang3.tuple.Pair;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.File;
import java.io.InputStream;
import java.io.Reader;
import java.net.URL;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;

/**
 * JSON related utilities
 *
 * @author Yinon Avraham.
 */
abstract class MapperUtilsBase {

    private final ObjectMapper mapper;

    protected MapperUtilsBase(ObjectMapper mapper) {
        this.mapper = mapper;
    }

    /**
     * Convert the given value to a JSON in string format
     *
     * @param value the value to convert
     * @return the result JSON string
     */
    @Nonnull
    public String valueToString(@Nullable Object value) {
        return valueToString(value, false);
    }

    /**
     * Convert the given value to a JSON in string format
     *
     * @param value       the value to convert
     * @param prettyPrint whether to return the string fabulous
     * @return the result JSON string
     */
    @Nonnull
    public String valueToString(@Nullable Object value, boolean prettyPrint) {
        return unchecked(() -> prettyPrint ?
                mapper.writerWithDefaultPrettyPrinter().writeValueAsString(value) :
                mapper.writeValueAsString(value)
        );
    }

    /**
     * Convert the given bean to a JSON tree
     *
     * @param value the value to convert
     * @return the result JSON tree
     */
    @Nonnull
    public JsonNode valueToTree(@Nullable Object value) {
        return unchecked(() -> mapper.valueToTree(value));
    }


    /**
     * Convert the given json to a JSON tree
     *
     * @param value the json to convert
     * @return the result JSON tree
     */
    @Nonnull
    public JsonNode readTree(@Nullable String value) {
        return unchecked(() -> mapper.readTree(value));
    }

    @Nonnull
    public JsonNode readTree(@Nullable File value) {
        return unchecked(() -> mapper.readTree(value));
    }

    @Nonnull
    public JsonNode readTree(@Nullable byte[] value) {
        return unchecked(() -> mapper.readTree(value));
    }

    /**
     * Create new Object to serialize
     */
    public ObjectNode createObjectNode() {
        return mapper.createObjectNode();
    }

    /**
     * Convert the given value to byte array
     *
     * @param value the value to convert
     * @return the result JSON string
     */
    public byte[] valueToByteArray(Object value) {
        return unchecked(() -> mapper.writeValueAsBytes(value));
    }


    public String writeValueAsString(Object value) {
        return unchecked(() -> mapper.writeValueAsString(value));
    }

    public void writeValue(File file, Object value) {
        unchecked(() -> {
            mapper.writeValue(file, value);
            return null;
        });
    }

    /**
     * Convert a string in the mapper format to a value in a given type
     *
     * @param object the string to read
     * @param clazz  the class type of the requested result value
     * @param <T>    the type of the result value
     * @return the result value
     */
    @Nonnull
    public <T> T readValue(@Nonnull String object, @Nonnull Class<T> clazz) {
        return unchecked(() -> mapper.readValue(object, clazz));
    }

    /**
     * Convert a stream in the mapper format to a value in a given type
     *
     * @param stream the stream to read
     * @param clazz  the class type of the requested result value
     * @param <T>    the type of the result value
     * @return the result value
     */
    @Nonnull
    public <T> T readValue(@Nonnull InputStream stream, @Nonnull Class<T> clazz) {
        return unchecked(() -> mapper.readValue(stream, clazz));
    }

    /**
     * Convert a string in the mapper format to a value in a given type
     *
     * @param json  the string to read
     * @param clazz the class type of the requested result value
     * @return the result value
     */
    @Nonnull
    public <T> T readValue(String json, TypeReference<T> clazz) {
        return unchecked(() -> mapper.readValue(json, clazz));
    }

    @Nonnull
    public <T> T readValueRef(String json, TypeReference<T> clazz) {
        return unchecked(() -> mapper.readValue(json, clazz));
    }

    @Nonnull
    public <T> T readValueRef(File file, TypeReference<T> clazz) {
        return unchecked(() -> mapper.readValue(file, clazz));
    }

    /**
     * Convert a JSON provided by a reader to a value in a given type
     *
     * @param reader the reader providing the JSON content to read
     * @param clazz  the class type of the requested result value
     * @param <T>    the type of the result value
     * @return the result value
     */
    @Nonnull
    public <T> T readValue(@Nonnull Reader reader, @Nonnull Class<T> clazz) {
        return unchecked(() -> mapper.readValue(reader, clazz));
    }

    /**
     * Clone the object with Mapper Serialization
     */
    public <T> T clone(@Nonnull T object, @Nonnull Class<T> clazz) {
        return readValue(valueToByteArray(object), clazz);
    }

    /**
     * Convert a JSON provided by a file to a value in a given type
     *
     * @param file  the file with the JSON content to read
     * @param clazz the class type of the requested result value
     * @param <T>   the type of the result value
     * @return the result value
     */
    @Nonnull
    public <T> T readValue(@Nonnull File file, @Nonnull Class<T> clazz) {
        return unchecked(() -> mapper.readValue(file, clazz));
    }

    /**
     * Convert a value provided by a url to a value in a given type
     *
     * @param url   the file with the JSON content to read
     * @param clazz the class type of the requested result value
     * @param <T>   the type of the result value
     * @return the result value
     */
    @Nonnull
    public <T> T readValue(@Nonnull URL url, @Nonnull Class<T> clazz) {
        return unchecked(() -> mapper.readValue(url, clazz));
    }

    /**
     * Convert a JSON provided by a byte array to a value in a given type
     *
     * @param bytes the bytes with the JSON content to read
     * @param clazz the class type of the requested result value
     * @param <T>   the type of the result value
     * @return the result value
     */
    @Nonnull
    public <T> T readValue(@Nonnull byte[] bytes, @Nonnull Class<T> clazz) {
        return unchecked(() -> mapper.readValue(bytes, clazz));
    }

    /**
     * Casts/converts a given value to {@link Long}
     *
     * @param value the value to convert. Supported input types: {@link Long} (no-op), {@link Integer}, {@link String}.
     * @return a {@link Long} value or <code>null</code> if the given value was <code>null</code>.
     * @throws IllegalArgumentException if the given value type is not supported
     */
    @Nullable
    public Long asLong(@Nullable Object value) {
        if (value == null) {
            return null;
        }
        if (value instanceof Long) {
            return (Long) value;
        }
        if (value instanceof Integer) {
            return ((Integer) value).longValue();
        }
        if (value instanceof String) {
            return Long.valueOf((String) value);
        }
        throw new IllegalArgumentException("Unexpected value type: " + value.getClass());
    }

    private <T> T unchecked(Callable<T> callable) {
        try {
            return callable.call();
        } catch (Exception e) {
            throw new JsonParsingException(e);
        }
    }

    public String merge(String original, String additionalData) {
        Map originalMap = readValue(original, Map.class);
        Map additionalDataMap = readValue(additionalData, Map.class);

        return valueToString(merge(originalMap, additionalDataMap));
    }

    public static Map<String, Object> merge(Map<String, Object> original, Map<String, Object> additionalData) {
        Set<String> allKeys = new HashSet<>(original.keySet());
        allKeys.addAll(additionalData.keySet());
        return allKeys.stream()
                .map(key -> {
                    Object prevValue = original.get(key);
                    if (additionalData.containsKey(key)) {
                        Object newValue = additionalData.get(key);
                        if (newValue == null) {
                            return Pair.of(key, null);
                        }

                        if (newValue instanceof Map && prevValue instanceof Map) {
                            return Pair.of(key, merge((Map) prevValue, (Map) newValue));
                        }

                        return Pair.of(key, newValue);
                    }

                    return Pair.of(key, prevValue);
                })
                .filter(pair -> pair.getValue() != null)
                .collect(Collectors.toMap(Pair::getLeft, Pair::getRight));
    }


    public <T> List<T> jsonStringToObjectArray(String content, Class<T> clazz) {
        return unchecked(
                () -> mapper.readValue(content, mapper.getTypeFactory().constructCollectionType(List.class, clazz)));
    }
}
