package org.jfrog.common;

import com.google.common.collect.Lists;
import org.apache.commons.io.FilenameUtils;
import org.codehaus.plexus.util.IOUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;

/**
 * @author Noam Shemesh
 */
public abstract class FileUtils {
    private static final Logger log = LoggerFactory.getLogger(FileUtils.class);

    private FileUtils() {
    }

    public static final String TIMESTAMP_FORMAT = "yyyyMMddHHmmss";
    public static final String TIMESTAMP_PLACEHOLDER = "${date}";
    private static final int TEMP_DIR_CREATE_ATTEMPTS = 10;
    private static final long MEGABYTE = 1024L * 1024L;

    public static void renameFileToTemplate(File srcFile, String nameTemplate) throws IOException {
        if (srcFile.exists()) {
            Path path = srcFile.toPath();
            String date = ClockUtils.getFormattedDateTimeNow(TIMESTAMP_FORMAT);
            String filename = nameTemplate.replaceAll(Pattern.quote(TIMESTAMP_PLACEHOLDER), date);
            Path targetPath = path.getParent().resolve(filename);
            log.debug("Renaming file '{}' to '{}'", srcFile, filename);
            Files.move(path, targetPath, StandardCopyOption.ATOMIC_MOVE);
        }
    }

    /**
     * Writes the content to the given file while maintaining a rolling file policy.<br> The file's last-modified
     * timestamp will be appended to each file that is rolled.<br>
     *
     * @param content    Content to write to the file
     * @param targetFile Target file
     * @param maxFiles   maximum accepted files. Must be positive
     */
    public static void writeContentToRollingFile(String content, File targetFile, int maxFiles) throws IOException {
        if (maxFiles <= 0) {
            throw new IllegalArgumentException("Maximum files must be positive");
        }

        if (!targetFile.exists()) {
            Files.createDirectories(targetFile.getParentFile().toPath());
            targetFile.createNewFile();
        } else {
            File parentDir = targetFile.getParentFile();

            String fileNameWithNoExtension = FilenameUtils.removeExtension(targetFile.getName());
            String fileExtension = FilenameUtils.getExtension(targetFile.getName());
            String newTargetFileName = fileNameWithNoExtension + "." + targetFile.lastModified() + "." + fileExtension;

            synchronized (targetFile.getPath().intern()) {
                org.apache.commons.io.FileUtils.copyFile(targetFile, new File(parentDir, newTargetFileName));
            }

            List<File> existingFileList = Lists.newArrayList(parentDir.listFiles((dir, name) ->
                    name.startsWith(fileNameWithNoExtension) && name.endsWith(fileExtension)));

            cleanRollingFiles(existingFileList, maxFiles);
        }

        org.apache.commons.io.FileUtils.writeStringToFile(targetFile, content);
    }

    /**
     * Removes the last files until only {maxFiles} are left
     *
     * @param files    - list of files to clean from
     * @param maxFiles - the amount of files to leave Must be positive.
     */
    public static void cleanRollingFiles(List<File> files, int maxFiles) {

        if (maxFiles <= 0) {
            throw new IllegalArgumentException("Maximum files must be positive");
        }

        Collections.sort(files);

        while (files.size() > maxFiles) {
            File toRemove = files.remove(0);
            org.apache.commons.io.FileUtils.deleteQuietly(toRemove);
        }
    }

    /**
     * Tries to delete the secret db.properties file if it exists,
     * throwing a runtime exception if the file exists but can't be deleted
     *
     * @param absoluteFilePath - file path
     */
    public static void handleSecretFileRemovalIfExist(File absoluteFilePath) {
        if (absoluteFilePath.exists()) {
            log.info("Secret db.properties at {} detected. Overriding all other db properties and attempting to delete",
                    absoluteFilePath.toPath());
            boolean deleted = absoluteFilePath.delete();
            if (!deleted) {
                log.error("Failed to delete Secrets file. Aborting");
                throw new RuntimeException("Failed to delete Secrets file.");
            } else {
                log.info("Secrets file deleted successfully");
            }
        }
    }

    /**
     * write byte array to file
     *
     * @param filename file name
     * @param content  byte array
     */
    public static void writeFile(String filename, byte[] content) {
        File file = new File(filename);
        FileOutputStream fop = null;
        try {
            /// check if file exist
            if (!file.exists()) {
                if (file.createNewFile()) {
                    fop = new FileOutputStream(file);
                    fop.write(content);
                    fop.flush();
                }
            }
        } catch (Exception e) {
            throw new IllegalStateException("Cannot write file to temporary directory");
        } finally {
            IOUtil.close(fop);
        }
    }


    /**
     * copy input stream to file
     *
     * @param in   - input stream
     * @param file - files
     */
    public static void copyInputStreamToFile(InputStream in, File file) {
        OutputStream out = null;
        try {
            out = new FileOutputStream(file);
            byte[] buf = new byte[1024];
            int len;
            while ((len = in.read(buf)) > 0) {
                out.write(buf, 0, len);
            }
        } catch (Exception e) {
            throw new IllegalStateException("Cannot copy file input stream");
        } finally {
            IOUtil.close(out);
            IOUtil.close(in);
        }
    }

    public static long bytesToMB(long sizeInBytes) {
        return sizeInBytes / MEGABYTE;
    }

    /**
     * Create directory (if not already exist)
     *
     * @param dir a directory to be created
     */
    public static void createDirectory(File dir, boolean transitive) {
        if (!dir.exists()) {
            for (int i = 0; i < TEMP_DIR_CREATE_ATTEMPTS; i++) {
                boolean ok;
                if (transitive) {
                    ok = dir.mkdirs();
                } else {
                    ok = dir.mkdir();
                }
                if (ok) {
                    break;
                }
            }
            if (!dir.exists()) {
                throw new IllegalStateException("Cannot create temporary directory");
            }
        }
        dir.setWritable(true);
    }

    public static void createDirectory(File dir) {
        createDirectory(dir, false);
    }

    public static void writeToFileAtomic(Path fileAbsolutePath, byte[] content) throws IOException {
        if (fileAbsolutePath == null || !fileAbsolutePath.isAbsolute()) {
            throw new IOException("Bad file path" + fileAbsolutePath);
        }
        Path dir = fileAbsolutePath.getParent();
        if (!Files.isDirectory(dir)) {
            Files.createDirectories(dir);
        }
        Path tempPath = Paths.get(fileAbsolutePath + ".tmp");
        Files.createFile(tempPath);
        File tempFile = tempPath.toFile();
        Files.write(tempPath, content);
        tempFile.renameTo(fileAbsolutePath.toFile());
    }
}
