package org.jfrog.common;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import static java.util.Objects.isNull;

/**
 * @author Noam Shemesh
 */
public abstract class ConfigArgUtils {
    private static final Logger log = LoggerFactory.getLogger(ConfigArgUtils.class);

    private ConfigArgUtils() {
    }

    public static Long requirePositive(Long value, String fieldName) {
        if (value == null) {
            return null;
        }

        return ArgUtils.requirePositive(value, getMessage(fieldName, value));
    }

    public static Integer requirePositive(Integer value, String fieldName) {
        if (value == null) {
            return null;
        }

        return ArgUtils.requirePositive(value, getMessage(fieldName, value));
    }

    /**
     * Require that all string in a given list are legal regular expressions
     *
     * @param value     list of regular expressions to check
     * @param fieldName name of field to require
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static List<String> requireLegalPatterns(List<String> value, String fieldName) {
        if (isNull(value) || value.isEmpty()) {
            return value;
        }
        return ArgUtils.requireSatisfies(value, (patterns -> {
            try {
                patterns.forEach(Pattern::compile);
                return true;
            } catch (PatternSyntaxException e) {
                log.debug("failed to compile regex. ", e);
                return false;
            }
        }), fieldName + " contains an illegal regular expression");
    }

    private static String getMessage(String fieldName, Number value) {
        return "'" + fieldName + "' must be a positive number. Got " + value;
    }

    public static Integer getSystemPropertyOrValue(String systemProp, Integer defaultValue) {
        return getSystemPropertyOrValue(systemProp, Integer::valueOf, defaultValue);
    }

    public static Long getSystemPropertyOrValue(String systemProp, Long defaultValue) {
        return getSystemPropertyOrValue(systemProp, Long::valueOf, defaultValue);
    }

    public static Boolean getSystemPropertyOrValue(String systemProp, Boolean defaultValue) {
        return getSystemPropertyOrValue(systemProp, Boolean::valueOf, defaultValue);
    }

    public static String getSystemPropertyOrValue(String systemProp, String defaultValue) {
        return getSystemPropertyOrValue(systemProp, Function.identity(), defaultValue);
    }

    private static <T> T getSystemPropertyOrValue(String sysProp, Function<String, T> valueOf, T defaultValue) {
        if (System.getProperties().containsKey(sysProp)) {
            try {
                return valueOf.apply(System.getProperty(sysProp));
            } catch (Exception e) {
                //do not log here, as is called before the log is ready
            }
        }
        return defaultValue;
    }

    public static Integer requireNonNegative(Integer value, String fieldName) {
        if (value == null) {
            return null;
        }

        return ArgUtils.requireNonNegative(value, getMessage(fieldName, value));
    }

    public static Float requirePositive(Float value, String fieldName) {
        if (value == null) {
            return null;
        }

        return ArgUtils.requirePositive(value, getMessage(fieldName, value));
    }

    public static <T> T requireSatisfies(Predicate<T> predicate, T value, String message) {
        if (value == null) {
            return null;
        }

        return ArgUtils.requireSatisfies(value, predicate, message);
    }

    @SuppressWarnings("unchecked")
    public static <T, R extends T> List<R> requireType(@Nonnull List<? extends T> entities,
            Class<R> clazz, String fieldName) {
        return requireType(entities, clazz, List.class, fieldName);
    }

    @SuppressWarnings("unchecked")
    public static <T, R extends T> Set<R> requireType(@Nonnull Set<? extends T> entities,
            Class<R> clazz, String fieldName) {
        return requireType(entities, clazz, Set.class, fieldName);
    }

    private static <T, R extends T, C extends Collection<? extends T>, RC extends Collection<R>> RC requireType(
            C entities, Class<R> clazz, Class<RC> collectionClazz, String fieldName) {
        if (entities.stream().anyMatch(e -> !clazz.isInstance(e))) {
            throw new IllegalStateException("All " + fieldName + " entities must be of instance " + clazz.getName());
        }

        return collectionClazz.cast(entities);
    }
}
