package org.jfrog.common;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * @author maximy, uriahl & yossis <3
 */
public class BiDirectionalLazyCacheImpl<K, V> extends LazyCacheImpl<K, V> implements BiDirectionalLazyCache<K, V> {
    private static final Logger log = LoggerFactory.getLogger(BiDirectionalLazyCacheImpl.class);

    private final Cache<V, K> invertedCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).build();
    private final Cache<V, K> invertedCacheDummy = CacheBuilder.newBuilder().maximumSize(0).build();

    private final Function<V, K> fetchInvertedFunction;
    private final Function<List<V>, Map<V, K>> fetchManyInverted;

    public BiDirectionalLazyCacheImpl(Function<K, V> fetchFunction, Function<V, K> fetchInvertedFunction,
            Function<List<K>, Map<K, V>> fetchMany, Function<List<V>, Map<V, K>> fetchManyInverted,
            Supplier<Map<K, V>> fetchAll) {
        super(fetchFunction, fetchMany, fetchAll);
        this.fetchInvertedFunction = fetchInvertedFunction;
        this.fetchManyInverted = fetchManyInverted;
    }

    @Override
    public Optional<V> value(K lookup) {
        Optional<V> value = super.value(lookup);
        value.ifPresent(v -> getInvertedCache().put(v, lookup));
        return value;
    }

    @Override
    public Optional<K> invertedValue(V lookup) {
        K value = getInvertedCache().getIfPresent(lookup);
        if (value == null) {
            value = fetchInvertedFunction.apply(lookup);
            if (value != null) {
                // minor optimization - populate both caches
                getCache().put(value, lookup);
                getInvertedCache().put(lookup, value);
            }
        }
        return Optional.ofNullable(value);

    }

    @Override
    public Map<K, Optional<V>> getValues(Collection<K> keys) {
        Map<K, Optional<V>> values = super.getValues(keys);
        values.forEach((key, optional) -> optional.ifPresent(value -> getInvertedCache().put(value, key)));
        return values;
    }

    @Override
    public Map<V, Optional<K>> getInvertedValues(Collection<V> keys) {
        Map<V, K> allCached = getAllInverted(keys);
        List<V> missing = getMissingInInvertedCache(keys, allCached);

        Map<V, K> fetched = fetchManyInverted.apply(missing);

        populateInvertedCacheWithMissingElements(fetched);
        return getBestVersion(keys, allCached, fetched);
    }

    @Override
    public void putAll(Map<K, V> entries) {
        entries.forEach((k, v) -> {
            getCache().put(k, v);
            getInvertedCache().put(v, k);
        });
    }

    @Override
    public void limitCachingToCurrentThread() {
        exclusiveCacheLock.lock();
        log.debug("Thread '{}' (id: '{}') acquired lock on bi-directional cache", Thread.currentThread().getName(),
                Thread.currentThread().getId());
        invalidateAll();
        log.info("Cache disabled!");
    }

    @Override
    public void invalidateAll() {
        getCache().invalidateAll();
        getInvertedCache().invalidateAll();
    }

    @Override
    public boolean invalidate(K k) {
        log.debug("Invalidating cache entry with key '{}'", k);
        V v = getCache().getIfPresent(k);
        if (v != null) {
            getInvertedCache().invalidate(v);
            getCache().invalidate(k);
        }
        return v != null;
    }

    @Override
    public void invalidate(Collection<K> keys) {
        keys.forEach(this::invalidate);
    }

    @Override
    public void removeCachingLimit(Class clazz) {
        invalidateAll();
        exclusiveCacheLock.unlock();
        log.info("Enabled cache for '{}'", clazz.getSimpleName());
    }

    @Override
    public LazyCache<K, V> prefetch() {
        invalidateAll();
        putAll(fetchAll.get());
        return this;
    }

    private void populateInvertedCacheWithMissingElements(Map<V, K> fetched) {
        fetched.forEach((key, value) -> {
            getInvertedCache().put(key, value);
            getCache().put(value, key);
        });
    }

    private List<V> getMissingInInvertedCache(Collection<V> keys, Map<V, K> allCached) {
        return keys.stream()
                .filter(s -> Objects.isNull(allCached.get(s)))
                .collect(Collectors.toList());
    }

    private Map<V, K> getAllInverted(Collection<V> keys) {
        return getInvertedCache().getAllPresent(keys);
    }

    private Cache<V, K> getInvertedCache() {
        return isCacheEnabled() || exclusiveCacheLock.isHeldByCurrentThread() ? invertedCache : invertedCacheDummy;
    }

    public long getInvertedCacheSize() {
        return getInvertedCache().size();
    }
}
