package org.jfrog.common.opentracing;

import io.jaegertracing.internal.JaegerSpan;
import io.jaegertracing.internal.JaegerTracer;
import io.jaegertracing.internal.samplers.SamplingStatus;
import io.jaegertracing.spi.Reporter;
import io.jaegertracing.spi.Sampler;

import java.util.HashMap;

/**
 * This project is migrating to OpenTracer.
 * The first stage it to add an id to logs so that distributed logs can be combined.
 * We are using a standard Open Tracing library so that later we can give fuller support with minimal changes.
 * <p>
 * We currently support a QUIET tracer, i.e. does not send data to some server (does not support telemetrics).
 * <p>
 * This is a bit different to the standard suggested by the Open Tracing documentation.
 * <p>
 * the standard recommends something like:
 * Configuration.SamplerConfiguration samplerConfig = Configuration.SamplerConfiguration.fromEnv().withType("const").withParam(0);
 * Configuration.ReporterConfiguration reporterConfig = Configuration.ReporterConfiguration.fromEnv().withLogSpans(true);
 * Configuration config = new Configuration("Access").withSampler(samplerConfig).withReporter(reporterConfig);
 * return config.getTracer();
 */
public class OpenTracerBuilder {
    //This is a factory that does not contain state, so no visible constructor provided
    private OpenTracerBuilder() {}

    public static JaegerTracer getQuietJaegerTracer(String tracerName) {
        return new JaegerTracer.Builder(tracerName)
                .withReporter(new EmptyReporter())
                .withSampler(new EmptySampler())
                .build();
    }

    static class EmptyReporter implements Reporter {

        @Override
        public void report(JaegerSpan jaegerSpan) {
            //this is a quiet reporter, it does not report anything
        }

        @Override
        public void close() {
            //since we do not have any resources or open handles we do not need to do anything here
        }
    }

    static class EmptySampler implements Sampler {
        @Override
        public SamplingStatus sample(String s, long l) {
            //need a real Map, since JaegerTracer puts tags there.
            return SamplingStatus.of(false, new HashMap<>());
        }

        @Override
        public void close() {
            //since we do not have any resources or open handles we do not need to do anything here
        }
    }
}

