package org.jfrog.common;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static java.util.Objects.isNull;

/**
 * @author maximy, uriahl & yossis <3
 */
public class LazyCacheImpl<K, V> implements LazyCache<K, V> {
    private static final Logger log = LoggerFactory.getLogger(LazyCacheImpl.class);

    private final Cache<K, V> cache;
    private final Cache<K, V> cacheDummy = CacheBuilder.newBuilder().maximumSize(0).build();

    private final Function<K, V> fetchFunction;
    private final Function<List<K>, Map<K, V>> fetchMany;
    final Supplier<Map<K, V>> fetchAll;

    final ReentrantLock exclusiveCacheLock = new ReentrantLock();

    public LazyCacheImpl(Function<K, V> fetchFunction, Function<List<K>, Map<K, V>> fetchMany,
            Supplier<Map<K, V>> fetchAll) {
        this(fetchFunction, fetchMany, fetchAll, 5);
    }

    public LazyCacheImpl(Function<K, V> fetchFunction, Function<List<K>, Map<K, V>> fetchMany,
            Supplier<Map<K, V>> fetchAll, int expireInMinutes) {
        this.fetchFunction = fetchFunction;
        this.fetchMany = fetchMany;
        this.fetchAll = fetchAll;
        this.cache = CacheBuilder.newBuilder().expireAfterWrite(expireInMinutes, TimeUnit.MINUTES).build();
    }

    @Override
    public Optional<V> value(K lookup) {
        V value = getCache().getIfPresent(lookup);
        if (value == null) {
            value = fetchFunction.apply(lookup);
            if (value != null) {
                getCache().put(lookup, value);
            }
        }
        return Optional.ofNullable(value);
    }

    @Override
    public V valueFromCache(K lookup) {
        return getCache().getIfPresent(lookup);
    }

    @Override
    public Map<K, Optional<V>> getValues(Collection<K> keys) {
        if (keys == null || keys.isEmpty()) {
            return Maps.newHashMap();
        }
        Map<K, V> allCached = getAllPresent(keys);
        List<K> missing = getMissingInCache(keys, allCached);

        Map<K, V> fetched = fetchMany.apply(missing);

        this.putAll(fetched);
        return getBestVersion(keys, allCached, fetched);
    }

    public V getValues(K lookup) {
        return getCache().getIfPresent(lookup);
    }

    public void put(K key, V value) {
        getCache().put(key, value);
    }

    @Override
    public void putAll(Map<K, V> entries) {
        entries.forEach((k, v) -> getCache().put(k, v));
    }

    @Override
    public void limitCachingToCurrentThread() {
        exclusiveCacheLock.lock();
        log.debug("Thread '{}' (id: '{}') acquired lock on bi-directional cache", Thread.currentThread().getName(),
                Thread.currentThread().getId());
        invalidateAll();
        log.info("Cache disabled!");
    }

    @Override
    public void invalidateAll() {
        getCache().invalidateAll();
    }

    @Override
    public boolean invalidate(K k) {
        log.debug("Invalidating cache entry with key '{}'", k);
        V v = getCache().getIfPresent(k);
        if (v != null) {
            getCache().invalidate(k);
        }
        return v != null;
    }

    @Override
    public void invalidate(Collection<K> keys) {
        getCache().invalidateAll(keys);
    }

    @Override
    public void removeCachingLimit(Class clazz) {
        invalidateAll();
        exclusiveCacheLock.unlock();
        log.info("Enabled cache for '{}'", clazz.getSimpleName());
    }

    @Override
    public LazyCache<K, V> prefetch() {
        invalidateAll();
        Map<K, V> entries = fetchAll.get();
        putAll(entries);
        return this;
    }

    @Override
    public Map<K, V> asMap() {
        return cache.asMap();
    }

    <T, U> Map<T, Optional<U>> getBestVersion(Collection<T> search,
            Map<T, U> cached, Map<T, U> fetched) {
        return search.stream().collect(Collectors.toMap(Function.identity(),
                g -> {
                    if (fetched.containsKey(g)) {
                        return Optional.of(fetched.get(g));
                    } else {
                        return Optional.ofNullable(cached.get(g));
                    }
                }));
    }

    private List<K> getMissingInCache(Collection<K> keys, Map<K, V> allCached) {
        return keys.stream()
                .filter(l -> isNull(allCached.get(l)))
                .collect(Collectors.toList());
    }

    private Map<K, V> getAllPresent(Collection<K> keys) {
        return getCache().getAllPresent(keys);
    }

    Cache<K, V> getCache() {
        return isCacheEnabled() || exclusiveCacheLock.isHeldByCurrentThread() ? cache : cacheDummy;
    }

    boolean isCacheEnabled() {
        return !exclusiveCacheLock.isLocked();
    }

    public long getCacheSize() {
        return getCache().size();
    }
}
