package org.jfrog.common;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Yinon Avraham.
 */
public abstract class ArgUtils {

    private final static ObjectMapper objectMapper = new ObjectMapper();

    private ArgUtils() {
        // utility
    }

    /**
     * Require that the given string value matches the given pattern
     *
     * @param value   the value to check
     * @param pattern the pattern to match
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    @Nonnull
    public static String requireMatches(String value, @Nonnull Pattern pattern, String message) {
        return requireSatisfies(value, v -> v != null && pattern.matcher(v).matches(), message);
    }

    /**
     * Require that the given string value is not blank
     *
     * @param value   the value to check
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    @Nonnull
    public static String requireNonBlank(String value, String message) {
        return requireSatisfies(value, StringUtils::isNotBlank, message);
    }

    /**
     * Require that the given long value is a positive number (i.e. <code>value &gt; 0</code>)
     *
     * @param value   the value to check
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static long requirePositive(Long value, String message) {
        return requireSatisfies(value, v -> v != null && v > 0, message);
    }

    /**
     * Require that the given long value is a positive number (i.e. <code>value &gt; 0</code>)
     * @param value the value to check
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static float requirePositive(Float value, String message) {
        return requireSatisfies(value, v -> v != null && v > 0, message);
    }

    /**
     * Require that the given int value is a positive number (i.e. <code>value &gt; 0</code>)
     *
     * @param value   the value to check
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static int requirePositive(Integer value, String message) {
        return requireSatisfies(value, v -> v != null && v > 0, message);
    }

    /**
     * Require that the given long value is a non-negative number (i.e. <code>value &gt;= 0</code>)
     *
     * @param value   the value to check
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static long requireNonNegative(Long value, String message) {
        return requireSatisfies(value, v -> v != null && v >= 0, message);
    }

    /**
     * Require that the given int value is a non-negative number (i.e. <code>value &gt;= 0</code>)
     *
     * @param value   the value to check
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static int requireNonNegative(Integer value, String message) {
        return requireSatisfies(value, v -> v != null && v >= 0, message);
    }

    /**
     * Require that the given String value is either null or has length no greater than maxLength
     *
     * @param value     the value to check
     * @param maxLength
     * @param message   the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static String requireMaxLength(String value, int maxLength, String message) {
        return requireSatisfies(value, v -> v == null || v.length() <= maxLength, message);
    }

    /**
     * Require that the given String value is either null or is consisted of basic name characters
     * which are: uppercase and lowercase letters, digits, dot, hyphen, underscore
     * This is a security check used to prevent attacks
     *
     * @param value   the value to check
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static String requireBasicNameCharacters(String value, String message) {
        return requireSatisfies(value, v -> v == null || v.matches("[-a-zA-Z0-9._ ]*"), message);
    }

    public static String requireNullOrJson(String value, String message) {
        return requireSatisfies(value, ArgUtils::isNullOrJson, message);
    }

    public static boolean isNullOrJson(String str) {
        if (str == null) {
            return true;
        }
        if (str.isEmpty()) {
            return false;
        }
        try {
            return objectMapper.readTree(str).isObject();
        } catch (IOException e) {
            return false;
        }
    }

    /**
     * Require that the given value is not null (different from Objects.requireNonNull by throwing {@link IllegalArgumentException} instead of {@link NullPointerException}
     *
     * @param value   the value to check
     * @param message the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static <T> T requireNonNull(T value, String message) {
        return requireSatisfies(value, Objects::nonNull, message);
    }

    /**
     * Require that exactly one of the given values matches the given filter.
     *
     * @param values    the values to check
     * @param predicate the predicate to test
     * @param message   the error message
     * @return the single matching value
     * @throws IllegalArgumentException in case the requirement fails
     * @see #arrayOf(Object[])
     */
    public static <T> T requireExactlyOne(T[] values, Predicate<T> predicate, String message) {
        List<T> result = Stream.of(values).filter(predicate).collect(Collectors.toList());
        if (result.size() != 1) {
            throw new IllegalArgumentException(message + " (found " + result.size() + " matching values)");
        }
        return result.get(0);
    }

    /**
     * Require that the given value satisfies the given predicate
     *
     * @param value     the value to check
     * @param predicate the predicate to satisfy
     * @param message   the error message in case the requirement fails
     * @return the given value
     * @throws IllegalArgumentException in case the requirement fails
     */
    public static <V> V requireSatisfies(V value, Predicate<V> predicate, String message) {
        if (!predicate.test(value)) {
            throw new IllegalArgumentException(message);
        }
        return value;
    }

    @SafeVarargs
    public static <T> T[] arrayOf(T... values) {
        return values;
    }

    public static <T> boolean isObjectEquals(T o1, T o2, Collection<Function<T, Object>> valueExtractors) {
        return valueExtractors.stream()
                .allMatch(extractor -> areEqual(extractor.apply(o1), extractor.apply(o2)));
    }

    public static <T> boolean areEqual(T o1, T o2) {
        if (o1 == null) {
            return o2 == null;
        }
        if (o2 == null) {
            return false;
        }
        if (!o1.getClass().equals(o2.getClass())) {
            return false;
        }
        if (o1 instanceof Collection) {
            return ((Collection) o1).containsAll((Collection) o2) && ((Collection) o2).containsAll((Collection) o1);
        }
        return o1.equals(o2);
    }

    public static <T> T requireType(Object object, Class<T> clazz, String message) {
        if (!clazz.isInstance(object)) {
            throw new IllegalArgumentException(message);
        }

        return clazz.cast(object);
    }

    public static String requireHttpHttpsUrl(String value, String message) {
        return requireSatisfies(value, v -> {
            try {
                URL url = new URL(value);
                if (!"http".equals(url.getProtocol()) && !"https".equals(url.getProtocol())) {
                    return false;
                }
            } catch (MalformedURLException e) {
                return false;
            }
            return true;
        }, message);
    }
}
