/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2019 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.common.logging.logback.servlet;

import ch.qos.logback.classic.LoggerContext;
import org.apache.commons.lang.StringUtils;
import org.jfrog.common.logging.logback.LogbackContextConfigurator;

import javax.servlet.ServletContext;
import java.io.File;
import java.io.Serializable;

import static java.util.Objects.requireNonNull;

/**
 * @author Yinon Avraham
 */
public class LoggerConfigInfo implements Serializable {

    private final String contextId;
    private final File logbackConfigFile;

    public LoggerConfigInfo(String contextId, File logbackConfigFile) {
        this.contextId = requireNonNull(StringUtils.lowerCase(contextId), "contextId is required");
        this.logbackConfigFile = requireNonNull(logbackConfigFile, "logback config file is required");
    }

    protected final String getContextId() {
        return contextId;
    }

    final File getLogbackConfigFile() {
        return logbackConfigFile;
    }

    final LoggerContext configure(LoggerContext loggerContext) {
        LogbackContextConfigurator configurator = LogbackContextConfigurator.configure(loggerContext);
        configure(configurator);
        configurator.configSource(getLogbackConfigFile());
        return configurator.build();
    }

    /**
     * An extension point for any additional configuration, mainly logger context properties.
     * Note - the config source is anyway overridden using {@link #getLogbackConfigFile()}
     *
     * @param configurator the configurator to use
     */
    protected void configure(LogbackContextConfigurator configurator) {
        //by default - no additional configuration
    }

    public static String getContextId(ServletContext servletContext) {
        return StringUtils.removeStart(servletContext.getContextPath(), "/");
    }
}
