package org.jfrog.common.logging.logback.rolling;

import ch.qos.logback.core.CoreConstants;
import ch.qos.logback.core.rolling.FixedWindowRollingPolicy;
import ch.qos.logback.core.rolling.RollingPolicyBase;
import ch.qos.logback.core.rolling.TimeBasedRollingPolicy;
import ch.qos.logback.core.rolling.helper.CompressionMode;
import ch.qos.logback.core.rolling.helper.Compressor;
import ch.qos.logback.core.rolling.helper.FileNamePattern;
import ch.qos.logback.core.rolling.helper.RenameUtil;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.lang.reflect.Field;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.TimeZone;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * A fixed windows, size based, rolling policy that names archive files to include the date of the roll time in the
 * standard way defined in JFrog products. The format of an archive file is according to
 * {@link FixedWindowWithDateRollingPolicy#ARCHIVE_FILE_DATE_PATTERN}. <p/>
 * This implementation used async compression using similar methods to {@link TimeBasedRollingPolicy}.
 * For instance, for a log file named service.log, the following files are expected
 * (assuming gz compression is enabled):
 * <pre>
 *  service.log
 *  service-2019-09-05T12-34-07.345.log.gz
 *  service-2019-09-07T02-12-45.123.log.gz
 *  service-2019-09-12T06-05-22.635.log.gz
 * </pre>
 *
 * @author Yossi Shaul
 * @author Nir Baram
 */
public class FixedWindowWithDateRollingPolicy extends FixedWindowRollingPolicy {
    private static final TimeZone UTC_TIME_ZONE = TimeZone.getTimeZone("UTC");
    private static final String ARCHIVE_FILE_DATE_PATTERN = "yyyy-MM-dd'T'HH-mm-ss.SSS";
    private static final int MAX_WINDOW_SIZE = 200;

    private RenameUtil renameUtil = new RenameUtil();

    private Future compressionFuture;

    @SuppressWarnings("WeakerAccess")
    public FixedWindowWithDateRollingPolicy() {
        setMaxIndex(10);
    }

    @Override
    public void rollover() {
        cleanupOldFiles();
        renameOrCompress();
    }

    @Override
    public void stop() {
        if (!isStarted()) { return; }
        waitForAsynchronousJobsToStop();
        super.stop();
    }

    @Override
    protected int getMaxWindowSize() {
        return MAX_WINDOW_SIZE;
    }

    private void cleanupOldFiles() {
        deleteOldestArchiveFiles(fileNamePatternStr, getMaxIndex() - getMinIndex());
    }

    private void renameOrCompress() {
        String targetFileName = getTargetFileName(fileNamePatternStr, compressionMode);
        renameUtil.rename(getActiveFileName(), targetFileName);
        if (compressionMode == CompressionMode.GZ) {
            compressionFuture =
                    getCompressor().asyncCompress(targetFileName, patternToFileName(fileNamePatternStr), null);
        } else if (compressionMode == CompressionMode.ZIP) {
            compressionFuture = getCompressor().asyncCompress(targetFileName,
                    patternToFileName(fileNamePatternStr), getZipNameEntryPattern().convert(new Date()));
        }
    }

    String getTargetFileName(String pattern, CompressionMode compressionMode) {
        return CompressionMode.NONE.equals(compressionMode) ?
                patternToFileName(pattern) : patternToFileName(pattern) + System.nanoTime() + ".tmp";
    }

    private String patternToFileName(String pattern) {
        return patternToFileName(pattern, formatDate(System.currentTimeMillis()));
    }

    String patternToFileName(String pattern, String date) {
        return pattern.replace("%i", date);
    }

    String formatDate(long time) {
        SimpleDateFormat dateFormat = createDateFormat();
        return dateFormat.format(time);
    }

    boolean isMyArchiveFile(String filePath, String pattern) {
        String[] pattenParts = pattern.split("%i");
        if (pattenParts.length != 2) {
            return false;
        }
        boolean beginAndEndsCorrectly = filePath.startsWith(pattenParts[0]) && filePath.endsWith(pattenParts[1]);
        if (!beginAndEndsCorrectly) {
            return false;
        }

        String date = extractDatePart(filePath, pattenParts);
        if (!validDateLength(date)) {
            return false;
        }

        try {
            createDateFormat().parse(date);
        } catch (ParseException e) {
            return false;
        }
        return true;
    }

    private boolean validDateLength(String date) {
        return date.length() == ARCHIVE_FILE_DATE_PATTERN.length() - 2;
    }

    private String extractDatePart(String filePath, String[] pattenParts) {
        String centralPart = StringUtils.removeStart(filePath, pattenParts[0]);
        centralPart = StringUtils.removeEnd(centralPart, pattenParts[1]);
        return centralPart;
    }

    void deleteOldestArchiveFiles(String pattern, int maxFilesToKeep) {
        File logDir = new File(pattern).getParentFile();
        File[] archiveFiles = logDir.listFiles(curPath -> isMyArchiveFile(curPath.getAbsolutePath(), pattern));
        if (archiveFiles != null && archiveFiles.length >= maxFilesToKeep) {
            Arrays.stream(archiveFiles).map(File::getAbsolutePath).sorted()
                    .limit((long) archiveFiles.length - maxFilesToKeep)
                    .forEach(path -> FileUtils.deleteQuietly(new File(path)));
        }
    }


    private SimpleDateFormat createDateFormat() {
        SimpleDateFormat dateFormat = new SimpleDateFormat(ARCHIVE_FILE_DATE_PATTERN);
        dateFormat.setTimeZone(UTC_TIME_ZONE);
        return dateFormat;
    }

    private Compressor getCompressor() {
        try {
            Field field = FixedWindowRollingPolicy.class.getDeclaredField("compressor");
            field.setAccessible(true);
            Object compressor = field.get(this);
            field.setAccessible(false);
            return (Compressor) compressor;
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new IllegalStateException("Failed accessing logback compressor", e);
        }
    }

    private FileNamePattern getZipNameEntryPattern() {
        try {
            Field field = RollingPolicyBase.class.getDeclaredField("zipEntryFileNamePattern");
            field.setAccessible(true);
            Object zipPattern = field.get(this);
            field.setAccessible(false);
            return (FileNamePattern) zipPattern;
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new IllegalStateException("Failed accessing logback zip pattern", e);
        }
    }

    void waitForAsynchronousJobsToStop() {
        if ((Future<?>) compressionFuture != null) {
            try {
                ((Future<?>) compressionFuture)
                        .get(CoreConstants.SECONDS_TO_WAIT_FOR_COMPRESSION_JOBS, TimeUnit.SECONDS);
            } catch (TimeoutException e) {
                addError("Timeout while waiting for " + "compression" + " job to finish", e);
            } catch (Exception e) {
                addError("Unexpected exception while waiting for " + "compression" + " job to finish", e);
            }
        }
    }

}
