package org.jfrog.common.logging.logback;

import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.joran.JoranConfigurator;
import ch.qos.logback.core.joran.spi.JoranException;
import ch.qos.logback.core.util.StatusPrinter;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

/**
 * Configuration helper for logback logger context
 *
 * @author Yinon Avraham.
 */
public class LogbackContextConfigurator {

    private final LoggerContext loggerContext;
    private final JoranConfigurator joranConfigurator = new JoranConfigurator();
    private final Map<String, String> properties = new HashMap<>();
    private ConfigSource configSource = ConfigSource.NULL_CONFIG_SOURCE;

    private LogbackContextConfigurator(LoggerContext loggerContext) {
        this.loggerContext = loggerContext;
    }

    /**
     * Start configuring a given logger context
     *
     * @param loggerContext the logger context to configure
     * @return a new logger context configurator
     * @see #build()
     */
    public static LogbackContextConfigurator configure(LoggerContext loggerContext) {
        return new LogbackContextConfigurator(loggerContext);
    }

    /**
     * Add a custom property to the logger context so it can be resolved in the logger config.
     *
     * @param key   the property key
     * @param value the property value
     * @return this configurator
     */
    public LogbackContextConfigurator property(String key, String value) {
        properties.put(key, value);
        return this;
    }

    /**
     * set the config source for the logger
     *
     * @param file the file with the logger config source
     * @return this configurator
     */
    public LogbackContextConfigurator configSource(File file) {
        configSource = new ConfigSource(file);
        return this;
    }

    /**
     * set the config source for the logger
     *
     * @param filename the filename with the logger config source
     * @return this configurator
     */
    public LogbackContextConfigurator configSource(String filename) {
        configSource = new ConfigSource(filename);
        return this;
    }

    /**
     * set the config source for the logger
     *
     * @param url the url of the file with the logger config source
     * @return this configurator
     */
    public LogbackContextConfigurator configSource(URL url) {
        configSource = new ConfigSource(url);
        return this;
    }

    /**
     * set the config source for the logger
     *
     * @param inputStream the input stream with the logger config source
     * @return this configurator
     */
    public LogbackContextConfigurator configSource(InputStream inputStream) {
        configSource = new ConfigSource(inputStream);
        return this;
    }

    /**
     * Build the logger context using this configurator
     *
     * @return the configured logger context
     */
    public LoggerContext build() {
        try {
            loggerContext.stop();
            joranConfigurator.setContext(loggerContext);
            properties.forEach(loggerContext::putProperty);
            configSource.configure(joranConfigurator);
            StatusPrinter.printIfErrorsOccured(loggerContext);
        } catch (JoranException je) {
            StatusPrinter.print(loggerContext);
        }
        return loggerContext;
    }

    private static class ConfigSource {
        public static final ConfigSource NULL_CONFIG_SOURCE = new ConfigSource(null, null, null, null);

        private final File file;
        private final URL url;
        private final String filename;
        private final InputStream inputStream;

        public ConfigSource(File file) {
            this(file, null, null, null);
        }

        public ConfigSource(URL url) {
            this(null, url, null, null);
        }

        public ConfigSource(String filename) {
            this(null, null, filename, null);
        }

        public ConfigSource(InputStream inputStream) {
            this(null, null, null, inputStream);
        }

        private ConfigSource(File file, URL url, String filename, InputStream inputStream) {
            this.file = file;
            this.url = url;
            this.filename = filename;
            this.inputStream = inputStream;
        }

        public void configure(JoranConfigurator configurator) throws JoranException {
            if (file != null) {
                configurator.doConfigure(file);
            } else if (filename != null) {
                configurator.doConfigure(filename);
            } else if (url != null) {
                configurator.doConfigure(url);
            } else if (inputStream != null) {
                configurator.doConfigure(inputStream);
            } else {
                throw new IllegalStateException("At least one config source is required.");
            }
        }
    }

}
