package org.jfrog.common.audit.request;

import org.jfrog.common.ClockUtils;

/**
 * @author Yinon Avraham.
 */
public class HttpRequestLogEntry {

    private String method;
    private String path;
    private String query;
    private String requestUri;
    private String scheme;
    private String remoteAddress;
    private String username;
    private String protocol;
    private String userAgent;
    private int status;
    private Long requestContentLength;
    private Long responseContentLength;
    private Long duration;
    private Long startTime;

    public String getMethod() {
        return method;
    }

    /**
     * Set the method of the request
     *
     * @param method the method name (e.g. <code>"GET"</code>, <code>"POST"</code>, etc.)
     * @return this log entry
     */
    public HttpRequestLogEntry method(String method) {
        this.method = method;
        return this;
    }

    public String getPath() {
        return path;
    }

    /**
     * Set the path part from the request URI
     *
     * @param path the path (e.g. <code>"foo/bar/hello"</code>
     * @return this log entry
     */
    public HttpRequestLogEntry path(String path) {
        this.path = path;
        return this;
    }

    public String getQuery() {
        return query;
    }

    /**
     * Set the query part from the request URI
     *
     * @param query the query (e.g. <code>"param1=value1&amp;param2=value2"</code>)
     * @return this log entry
     */
    public HttpRequestLogEntry query(String query) {
        this.query = query;
        return this;
    }

    public String getRequestUri() {
        return requestUri;
    }

    /**
     * Set the request URI
     *
     * @param requestUri the request URI
     * @return this log entry
     */
    public HttpRequestLogEntry requestUri(String requestUri) {
        this.requestUri = requestUri;
        return this;
    }

    public String getScheme() {
        return scheme;
    }

    /**
     * Set the scheme of the request
     *
     * @param scheme the scheme of the request (e.g. <code>"http"</code>, <code>"https"</code>)
     * @return this log entry
     */
    public HttpRequestLogEntry scheme(String scheme) {
        this.scheme = scheme;
        return this;
    }

    public String getRemoteAddress() {
        return remoteAddress;
    }

    /**
     * Set the remote address the request originated from
     *
     * @param remoteAddress the remote address (i.e. IP)
     * @return this log entry
     */
    public HttpRequestLogEntry remoteAddress(String remoteAddress) {
        this.remoteAddress = remoteAddress;
        return this;
    }

    public String getUsername() {
        return username;
    }

    /**
     * Set the name of the authenticated user
     *
     * @param username the user name
     * @return this log entry
     */
    public HttpRequestLogEntry username(String username) {
        this.username = username;
        return this;
    }

    public String getProtocol() {
        return protocol;
    }

    /**
     * Set the protocol used for the request
     *
     * @param protocol the protocol, e.g. <code>"HTTP/1.1"</code>
     * @return this log entry
     */
    public HttpRequestLogEntry protocol(String protocol) {
        this.protocol = protocol;
        return this;
    }

    public String getUserAgent() {
        return userAgent;
    }

    /**
     * Set the user-agent which issued the request
     *
     * @param userAgent the user-agent
     * @return this log entry
     */
    public HttpRequestLogEntry userAgent(String userAgent) {
        this.userAgent = userAgent;
        return this;
    }

    public int getStatus() {
        return status;
    }

    /**
     * Set the response status
     *
     * @param status the HTTP status of the response
     * @return this log entry
     */
    public HttpRequestLogEntry status(int status) {
        this.status = status;
        return this;
    }

    public Long getRequestContentLength() {
        return requestContentLength;
    }

    /**
     * Set the content-length of the request
     *
     * @param contentLength the content-length
     * @return this log entry
     */
    public HttpRequestLogEntry requestContentLength(Long contentLength) {
        this.requestContentLength = contentLength;
        return this;
    }

    public Long getResponseContentLength() {
        return responseContentLength;
    }

    /**
     * Set the content-length of the response
     *
     * @param contentLength the content-length
     * @return this log entry
     */
    public HttpRequestLogEntry responseContentLength(Long contentLength) {
        this.responseContentLength = contentLength;
        return this;
    }

    /**
     * Get the duration the request took
     *
     * @return the duration in milliseconds, or <code>null</code> if the duration was not set
     */
    public Long getDuration() {
        return duration;
    }

    /**
     * Set the duration the request took
     *
     * @param duration the duration in milliseconds
     * @return this log entry
     */
    public HttpRequestLogEntry duration(Long duration) {
        this.duration = duration;
        return this;
    }

    /**
     * Start the the clock for this log entry, will be used to calculate the duration the request took
     *
     * @return this log entry
     * @see #stop()
     * @see #getDuration()
     */
    public HttpRequestLogEntry start() {
        this.startTime = ClockUtils.epochMillis();
        return this;
    }

    /**
     * Stop the the clock for this request log entry, calculate the duration the request took.
     * If {@link #start()} was not called on this entry, this is a no-op and the duration will not be calculated.
     *
     * @return this log entry
     * @see #start()
     * @see #getDuration()
     */
    public HttpRequestLogEntry stop() {
        if (startTime != null) {
            this.duration = ClockUtils.epochMillis() - startTime;
        }
        return this;
    }

    @Override
    public String toString() {
        return String.join(" ", scheme, method, path, "" + status);
    }
}
