/*
 *
 * Copyright 2019 JFrog Ltd. All rights reserved.
 * JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.common.logging.logback.servlet;

import ch.qos.logback.classic.ClassicConstants;
import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.selector.ContextSelector;
import ch.qos.logback.classic.util.ContextSelectorStaticBinder;
import org.jfrog.common.util.FileWatchDog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.util.Objects.requireNonNull;

/**
 * Logback config manager, responsible for configuring the logback context and for managing the logback configuration
 * file watch-dog.
 * <p>
 * <b>Common usage:</b><br>
 * Initialization:
 * <pre>
 * LoggerConfigInfo configInfo = ...
 * LogbackConfigManager configManager = new LogbackConfigManager(configInfo);
 * storeConfigManager(configManager); //store the config manager somewhere for later (e.g. application context)
 * configManager.configureLogbackContext();
 * configManager.startWatchDog(15000); //Check for refresh every 15 seconds (default: 30 seconds)
 * </pre>
 * Destruction:
 * <pre>
 * LogbackConfigManager configManager = restoreConfigManager();
 * configManager.destroy();
 * </pre>
 * <p>
 *
 * @author Yinon Avraham
 */
public class LogbackConfigManager {

    private final LoggerConfigInfo configInfo;
    private LogbackConfigWatchDog configWatchDog;
    private LoggerContext loggerContext;
    private LogbackNotifier logbackNotifier;

    public LogbackConfigManager(LoggerConfigInfo configInfo) {
        this.configInfo = requireNonNull(configInfo, "logger config info is required");
        logbackNotifier = new LogbackNotifier();
    }

    /**
     * Configure the logback context - handles both cases of with or without the custom {@link LogbackContextSelector}.
     */
    public void configureLogbackContext() {
        boolean selectorUsed = isSelectorUsed();
        LoggerContext context;
        if (selectorUsed) {
            LogbackContextSelector.bindConfig(configInfo);
            try {
                //This load should already use a context from the selector
                context = getOrInitLoggerContext();
            } finally {
                LogbackContextSelector.unbindConfig();
            }
        } else {
            context = getOrInitLoggerContext();
            context = configInfo.configure(context);
        }
        loggerContext = requireNonNull(context, "logger context is required");
    }

    public void startWatchDog(Long refreshInterval) {
        //Configure and start the watchdog
        configWatchDog = new LogbackConfigWatchDog(loggerContext, configInfo);
        if (refreshInterval != null) {
            configWatchDog.setDelay(refreshInterval);
        }
        configWatchDog.start();
    }

    public void registerObserverForLogbackChange(LogbackObserver observer) {
            this.logbackNotifier.register(observer);
    }

    private static LoggerContext getOrInitLoggerContext() {
        return (LoggerContext) LoggerFactory.getILoggerFactory();
    }

    public void relaxWebComponentLogLevel() {
        try {
            LoggerContext loggerContext = getOrInitLoggerContext();
            // Relax the log level of the WebComponent class.
            // It has a single warning message which should actually be in debug:
            // "A servlet request ... contains form parameters ... the request body has been consumed ...
            //  Only resource methods using @FormParam will work as expected. ..."
            loggerContext.getLogger("com.sun.jersey.spi.container.servlet.WebComponent").setLevel(Level.ERROR);
        } catch (Exception e) {
            //Ignore - this should not fail the application to the start
            System.err.println("Could not relax WebComponent log level. Continuing.");
        }
    }

    public void destroy() {
        ContextSelector selector = ContextSelectorStaticBinder.getSingleton().getContextSelector();
        selector.detachLoggerContext(configInfo.getContextId());
        destroyWatchDog();
    }

    private void destroyWatchDog() {
        if (configWatchDog != null) {
            configWatchDog.interrupt();
            configWatchDog.loggerContext.stop();
        }
    }

    private class LogbackConfigWatchDog extends FileWatchDog {

        private final Logger log = LoggerFactory.getLogger(LogbackConfigWatchDog.class);

        private LoggerContext loggerContext;
        private LoggerConfigInfo configInfo;
        private final boolean selectorUsed;

        LogbackConfigWatchDog(LoggerContext loggerContext, LoggerConfigInfo configInfo) {
            super(configInfo.getLogbackConfigFile(), false);
            setName("logback-watchdog");
            this.selectorUsed = isSelectorUsed();
            this.configInfo = configInfo;
            this.loggerContext = requireNonNull(loggerContext, "logger context is required");
            checkAndConfigure();
        }

        @Override
        protected void doOnChange() {
            if (selectorUsed) {
                // if the selector is used, then bind a new LoggerConfigInfo.
                // see JFW-1180
                bind(configInfo);
            }
            try {
                configInfo.configure(loggerContext);
                logbackNotifier.runJobs();
                //Log after re-config, since this class logger is constructed before config with the default warn level
                log.info("Reloaded logback config from: {}.", file.getAbsolutePath());
            } catch (Exception ex) {
                //Do nothing - error is already printed as part of LogbackContextConfigurator#build
            } finally {
                if (selectorUsed) {
                    unbind();
                }
            }
        }

        private void bind(LoggerConfigInfo configInfo) {
            LogbackContextSelector.bindConfig(configInfo);
        }

        private void unbind() {
            LogbackContextSelector.unbind();
        }
    }

    private boolean isSelectorUsed() {
        return System.getProperty(ClassicConstants.LOGBACK_CONTEXT_SELECTOR) != null;
    }

}
