/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.common.logging.logback.filters.contextaware.request;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.turbo.MatchingFilter;
import ch.qos.logback.core.spi.FilterReply;
import org.jfrog.common.logging.logback.filters.contextaware.*;
import org.slf4j.Marker;

/**
 * @author haims
 */
public class RequestAwareLogbackFilter extends MatchingFilter {

    public RequestAwareLogbackFilter(){
        onMatch = FilterReply.ACCEPT;
    }

    private ContextMatching contextMatching = new ContextMatching();
    private LoggerMatching loggerMatching = new LoggerMatching();


    private static ThreadLocal<Boolean> inEvaluation = new ThreadLocal<>();


    @Override
    public FilterReply decide(Marker marker, Logger logger, Level level, String format, Object[] params, Throwable t) {
        boolean initInEvaluation = false;
        FilterReply decision = FilterReply.NEUTRAL;
        try {
            if (inEvaluation == null){
                decision = FilterReply.NEUTRAL;
            } else {
                Boolean entered = inEvaluation.get();
                if (entered == null || !entered) {
                    inEvaluation.set(true);
                    initInEvaluation = true;
                    decision = evaluateEvent(logger, level);
                } else {
                    decision = FilterReply.NEUTRAL;
                }
            }
        } finally {
            if (initInEvaluation) {
                inEvaluation.set(false);
            }
        }
        return decision;
    }

    private FilterReply evaluateEvent(Logger logger, Level level) {
        try {
            LogbackContext requestContext = LogbackContextHolder.getContext();
            if (requestContext == null) {
                return FilterReply.NEUTRAL;
            }
            boolean loggerMatchingResult = loggerMatching.match(logger, level);
            if (loggerMatchingResult) {
                boolean contextResult = contextMatching.match(requestContext);
                if (contextResult) {
                    return onMatch;
                }
            }
            return onMismatch;
        } catch (Exception e) {
            return FilterReply.NEUTRAL;
        }
    }


    public void setLogger(LoggerMatcher loggerMatcher) {
        this.loggerMatching.addLoggerMatcher(loggerMatcher);
    }

    public void setFilter(String filterExpression) {
        this.contextMatching.addFilter(filterExpression);
    }

    public void setContextMatchers(ContextMatchers contextMatchers) {
        this.contextMatching = new ContextMatching(contextMatchers);
    }

}