package org.jfrog.common.audit.request;

import org.jfrog.common.logging.slf4j.MDCAutoRemovable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Closeable;

/**
 * Logs request details to the request logger.
 * The default format is as follows: <p>
 * <code>%date{yyyy-MM-dd'T'HH:mm:ss.SSSZ}|%X{req.remoteAddress}|%X{req.username}|%X{req.method}|%X{req.uri}|%X{req.status}|%X{req.responseContentLength}|%X{req.duration}|%X{req.userAgent}}</code>
 * <p> For instance: </p>
 * <code>2017-05-15T11:33:28.945+0300|0:0:0:0:0:0:0:1|admin|GET|https://localhost:8040/api/v1/cert/root|200|-1|166|curl/7.45.0</code>
 *
 * @author Yinon Avraham.
 */
public class RequestLogger {
    private static final Logger log = LoggerFactory.getLogger(RequestLogger.class);

    private static final String UNKNOWN = "UNKNOWN";
    private static final String NON_AUTH_USER = "anonymous";
    private static final String ZERO = "0";
    private static final String DURATION_NA = "-1";
    private static final String EMPTY = "";
    private static final String ROOT = "/";
    private static final String HTTP_GET_METHOD = "GET";

    public static void log(HttpRequestLogEntry logEntry) {
        try (Closeable autoRemovableKeys = new MDCAutoRemovable()
                .put("req.method", valueOrDefault(logEntry.getMethod(), UNKNOWN))
                .put("req.path", valueOrDefault(logEntry.getPath(), ROOT))
                .put("req.query", valueOrDefault(logEntry.getQuery(), EMPTY))
                .put("req.uri", valueOrDefault(logEntry.getRequestUri(), UNKNOWN))
                .put("req.scheme", valueOrDefault(logEntry.getScheme(), UNKNOWN))
                .put("req.protocol", valueOrDefault(logEntry.getProtocol(), UNKNOWN))
                .put("req.remoteAddress", valueOrDefault(logEntry.getRemoteAddress(), UNKNOWN))
                .put("req.userAgent", valueOrDefault(logEntry.getUserAgent(), UNKNOWN))
                .put("req.username", valueOrDefault(logEntry.getUsername(), NON_AUTH_USER))
                .put("req.status", positiveValueOrDefault(logEntry.getStatus(), ZERO))
                .put("req.contentLength", valueOrDefault(logEntry.getRequestContentLength(), UNKNOWN))
                .put("res.contentLength", valueOrDefault(logEntry.getResponseContentLength(), UNKNOWN))
                .put("contentLength", valueOrDefault(relevantContentLength(logEntry), UNKNOWN))
                .put("req.duration", valueOrDefault(logEntry.getDuration(), DURATION_NA))) {
            log.info(""); //No specific message, the appender's pattern should use the variables above.
        } catch (Exception e) {
            log.error("Failed to log http request entry: {}", e.toString());
            log.debug("Failed to log http request entry.", e);
        }
    }

    private static Long relevantContentLength(HttpRequestLogEntry logEntry) {
        return logEntry.getMethod().equalsIgnoreCase(HTTP_GET_METHOD) ? logEntry.getResponseContentLength()
                : logEntry.getRequestContentLength();
    }

    private static String valueOrDefault(String value, String defaultValue) {
        return value == null ? defaultValue : value;
    }

    private static String valueOrDefault(Long value, String defaultValue) {
        return value == null ? defaultValue : "" + value;
    }

    private static String positiveValueOrDefault(int value, String defaultValue) {
        return value > 0 ? "" + value : defaultValue;
    }

    private static String positiveValueOrDefault(long value, String defaultValue) {
        return value > 0 ? "" + value : defaultValue;
    }

    /**
     * @return true iff the request logger info level or finer is enabled (i.e., true if logs requests will be processed)
     */
    public static boolean isEnabled() {
        return log.isInfoEnabled();
    }
}
