package org.jfrog.common.logging.logback.servlet;

import org.jfrog.common.logging.BootstrapLogger;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

/**
 * A base implementation for a servlet context listener for configuring logback based on a config file.
 *
 * @see #createLoggerConfigInfo(ServletContext)
 * @author Yinon Avraham.
 */
public abstract class LogbackConfigListenerBase implements ServletContextListener {

    private static final String CTX_ATTR_LOGBACK_CONFIG_MANAGER = LogbackConfigListenerBase.class.getName() + ".ATTR_LOGBACK_CONFIG_MANAGER";

    @Override
    public void contextInitialized(ServletContextEvent event) {
        ServletContext servletContext = event.getServletContext();
        LoggerConfigInfo configInfo = createLoggerConfigInfo(servletContext);
        LogbackConfigManager configManager = new LogbackConfigManager(configInfo);
        servletContext.setAttribute(CTX_ATTR_LOGBACK_CONFIG_MANAGER, configManager);
        configManager.configureLogbackContext();
        configManager.startWatchDog(getRefreshInterval(servletContext));
        if (!isDevOrTest()) {
            configManager.relaxWebComponentLogLevel();
        }
    }

    private Long getRefreshInterval(ServletContext servletContext) {
        Long refreshInterval = null;
        String intervalString = servletContext.getInitParameter("logbackRefreshInterval");
        if (intervalString != null) {
            try {
                refreshInterval = Long.parseLong(intervalString);
            } catch (NumberFormatException e) {
                BootstrapLogger.error("Failed to parse logbackRefreshInterval. Using default log refresh interval.");
            }
        }
        return refreshInterval;
    }

    /**
     * Create the logger config info to use for configuring the logger context.
     */
    protected abstract LoggerConfigInfo createLoggerConfigInfo(ServletContext servletContext);

    protected boolean isDevOrTest() {
        return false;
    }

    @Override
    public void contextDestroyed(ServletContextEvent servletContextEvent) {
        ServletContext servletContext = servletContextEvent.getServletContext();
        LogbackConfigManager configManager = (LogbackConfigManager) servletContext.getAttribute(CTX_ATTR_LOGBACK_CONFIG_MANAGER);
        if (configManager != null) {
            configManager.destroy();
        }
    }
}
