/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.common.logging.logback.filters.contextaware;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.jayway.jsonpath.JsonPath;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import static org.jfrog.common.logging.logback.filters.contextaware.DataRetriever.NIL;

/**
 * @author haims
 */
public class BasicLogbackContext implements LogbackContext {
    private final Boolean useCache;
    private ObjectMapper mapper;
    private Map<String, Object> data = new HashMap<>();
    private Map<String, String> cachedValues = new HashMap<>();
    private Map<String, Supplier<Object>> fetchers = new HashMap<>();
    private List<DataRetriever> retrievers = new ArrayList<>();
    private static final String JSON_PATH_PREFIX = "$.";
    private Map<String, JsonPath> jsonPaths = new ConcurrentHashMap<>();

    public BasicLogbackContext(Boolean useCache) {
        this.useCache = useCache;
        mapper = new ObjectMapper();
    }

    public BasicLogbackContext() {
        this(true);
    }

    private Map getMapValue(Object value) {
        return value instanceof String ? null : mapper.convertValue(value, Map.class);
    }

    public void addDataRetriever(DataRetriever retriever) {
        retrievers.add(retriever);
    }

    public void setKeyValueFetcher(String key, Supplier<Object> supplier) {
        fetchers.put(key, supplier);
    }

    @Override
    public void put(String key, Object value) {
        data.put(key, value);
    }

    @Override
    public String get(String key) {
        if (useCache && cachedValues.containsKey(key))
        {
            String value = cachedValues.get(key);
            if (Objects.equals(value, NIL)) return null;
            return value;
        }
        String value = findFirstMatchingKey(key, null);
        if (useCache && value != null){
            cachedValues.put(key, value);
        }
        return value;
    }

    private String findFirstMatchingKey(String key, String suffix) {
        if (key == null || key.isEmpty()) {
            return null;
        }
        Object value = getExplicitValue(key);
        String calculatedValue = null;
        if (value ==  null) {
            int prefixIndex = key.lastIndexOf(".");
            if (prefixIndex > 0) {
                String newKey = key.substring(0, prefixIndex);
                String newSuffix = key.substring(prefixIndex + 1);
                if (!newKey.isEmpty()) {
                    calculatedValue = findFirstMatchingKey(newKey, newSuffix);
                }
            }
        } else if (!(value instanceof String)){
            if (suffix != null && !suffix.isEmpty()) {
                Map mapValue = getMapValue(value);
                if (mapValue == null) return NIL;
                String jsonPathKey = JSON_PATH_PREFIX + suffix;
                JsonPath jsonPath = this.jsonPaths.get(jsonPathKey);
                if (jsonPath == null) {
                    jsonPath = JsonPath.compile(jsonPathKey);
                    jsonPaths.put(jsonPathKey, jsonPath);
                }
                try {
                    calculatedValue = jsonPath.read(mapValue);
                } catch (Exception e) {
                    // leave value as null
                }
            }
            if (calculatedValue == null) {
                calculatedValue = NIL;
            }
        } else {
            calculatedValue = (String) value;
        }
        return calculatedValue;
    }

    private Object getExplicitValue(String key) {
        Object value = data.get(key);
        if (value == null){
            Supplier<Object> fetcher = fetchers.get(key);
            if (fetcher !=  null){
                try {
                    value = fetcher.get();
                } catch (Exception ignored){}
            }
        }
        if (value == null) {
            for (DataRetriever retriever : retrievers) {
                value = retriever.get(key);
            }
        }
        return value;
    }

    public void clear() {
        this.data.clear();
        this.fetchers.clear();
    }

    public void remove(String key) {
        this.data.remove(key);

    }

    @Override
    public void destroy() {

    }

    public void removeFetcher(String key) {
        this.fetchers.remove(key);
    }
}
