/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.common.logging;

import java.io.File;
import java.nio.file.Files;
import java.text.SimpleDateFormat;
import java.util.Date;

import static java.nio.file.StandardOpenOption.*;
import static java.util.Collections.singleton;

/**
 * @author Yoav Landman
 */
public abstract class BootstrapLogger {

    private static File logFile;
    private static String appName = "UNDEFINED";

    private BootstrapLogger() {
        // utility class
    }

    public static void info(String msg) {
        log("INFO ", msg);
    }

    public static void warn(String msg) {
        log("WARN ", msg);
    }

    public static void error(String msg) {
        log("ERROR", msg);
    }

    private static void log(String level, String msg) {
        String logMessage = formattedTimestamp() + " [" + appName + "] [" + level + "] " + msg;
        System.out.println(logMessage);
        writeToFile(logMessage);
    }

    private static String formattedTimestamp() {
        return new SimpleDateFormat("YYYY-MM-dd HH:mm:ss").format(new Date());
    }

    private static void writeToFile(String logMessage) {
        if (logFile != null) {
            try {
                Files.write(logFile.toPath(), singleton(logMessage), CREATE, WRITE, APPEND);
            } catch (Exception e) {
                //ignore
            }
        }
    }

    public static synchronized void init(String applicationName, File logDir) {
        appName = applicationName;
        logFile = null;
        if (logDir != null) {
            logFile = createNewLogFile(logDir);
        } else {
            warn("Could not initialize bootstrap log files in directory: " + logDir);
        }
    }

    private static File createNewLogFile(File logDir) {
        File file = null;
        try {
            if (logDir.exists() || logDir.mkdirs()) {
                file = new File(logDir, appName.toLowerCase() + ".bootstrap.log");
                if (file.exists() && !file.delete()) {
                    warn("Could not delete existing bootstrap log file: " + file);
                }
            }
        } catch (Exception e) {
            warn("Could not initialize bootstrap log files in directory '" + logDir + "': " + e);
        }
        if (file == null) {
            warn("Bootstrap log file was not initialized. Logs directory: " + logDir);
        }
        return file;
    }

    public static synchronized void cleanup() {
        File file = logFile;
        logFile = null;
        deleteFile(file);
    }

    private static void deleteFile(File file) {
        if (file != null && file.exists()) {
            try {
                if (!file.delete()) {
                    warn("Could not delete bootstrap log file: " + file);
                }
            } catch (Exception e) {
                warn("Could not delete bootstrap log file '" + file + "': " + e);
            }
        }
    }
}