/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.storage.common;

import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * A data structure which allows only one concurrent writer and many readers.
 *
 * @author Shay Yaakov
 */
public interface LockingMap<T> {

    /**
     * Adds a key to the map while locking other threads from locking the same key concurrently.
     *
     * @param key      The key to add
     * @param timeout  the maximum time to wait for the lock. Value of 0 doesn't wait
     * @param timeUnit the time unit of the {@code time} argument
     * @return {@code true} if the lock was acquired and {@code false}
     * if the waiting time elapsed before the lock was acquired
     * @throws InterruptedException if the current thread is interrupted
     *                              while acquiring the lock (and interruption of lock
     *                              acquisition is supported)
     */
    boolean tryToLock(T key, long timeout, TimeUnit timeUnit) throws InterruptedException;

    /**
     * Removes the path from the map and releases it's lock.
     *
     * @param key The key to remove
     */
    void unlock(T key);


    /**
     * Removes the path from the map and releases it's lock regardless the thread owner.
     *
     * @param key The key to remove
     */
    void forceUnlock(T key);

    /**
     * checks if the path is locked.
     *
     * @param key The key to remove
     */
    boolean isLocked(T key);

    /**
     * return the size of the locking map.
     */
    int size();

    /**
     * return the key set of the locking map.
     */
    Set<T> keySet();
}

