package org.jfrog.client.http;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.impl.conn.DefaultRoutePlanner;
import org.apache.http.impl.conn.DefaultSchemePortResolver;
import org.apache.http.protocol.HttpContext;
import org.jfrog.client.util.NoProxyHostsEvaluator;

import javax.annotation.Nullable;

/**
 * Implementation that uses the (optional) default configured {@param defaultHost}
 * if the request doesn't explicitly specify the host as part of the request query.
 * It also includes configuring (optional) proxy {@param proxyHost} for the
 * request and an option to bypass this proxy if targetHost is specified in the
 * list of {@param noProxyHosts}. localhost will bypass the proxy by default!
 *
 * NOTE 1: When using this routePlanner, you MAY NOT configure proxy in
 * the RequestConfig of the HttpBuilderBase
 *
 * NOTE 2: This solution allows configuring the above 3 parameters at ONCE.
 * Other alternatives to achieve that separately would be:
 * 1.   Extending SystemDefaultRoutePlanner, passing your own ProxySelector,
 *      passing defaultHost and override method determineRoute().
 * 2.   Using SystemDefaultRoutePlanner with useSystemParameters() for getting
 *      proxy and nonProxyHosts from system parameters. defaultHost should be
 *      passed to another extended DefaultRoutePlanner like the previous
 *      DefaultHostRoutePlanner
 */
@Slf4j
public class DefaultHostSpecificProxyRoutePlanner extends DefaultRoutePlanner {

    private final HttpHost defaultHost;
    private final HttpHost proxyHost;
    private final NoProxyHostsEvaluator noProxyHostsEvaluator;

    /**
     *
     * @param defaultHost used when targetHost is not explicitly
     *                    specified in the request (optional)
     * @param proxyHost proxy host if configured (optional)
     * @param noProxyHosts set of domains for which the proxy should not be consulted;
     *                     the contents is a comma-separated list of domain names,
     *                     with an optional :port part.
     *                     example: "cern.ch,ncsa.uiuc.edu,some.host:8080"
     */
    private DefaultHostSpecificProxyRoutePlanner(@Nullable HttpHost defaultHost,
            @Nullable HttpHost proxyHost, @Nullable String noProxyHosts) {
        super(DefaultSchemePortResolver.INSTANCE);
        this.defaultHost = defaultHost;
        this.proxyHost = proxyHost;
        this.noProxyHostsEvaluator = new NoProxyHostsEvaluator(noProxyHosts);
    }

    public static class Builder {

        private HttpHost defaultHost;
        private HttpHost proxyHost;
        private String noProxyHosts;

        public Builder defaultHost(HttpHost defaultHost) {
            this.defaultHost = defaultHost;
            return this;
        }

        public Builder proxyHost(HttpHost proxyHost) {
            this.proxyHost = proxyHost;
            return this;
        }

        public Builder noProxyHosts(String noProxyHosts){
            this.noProxyHosts = noProxyHosts;
            return this;
        }

        public DefaultHostSpecificProxyRoutePlanner build() {
            return new DefaultHostSpecificProxyRoutePlanner(
                    defaultHost, proxyHost, noProxyHosts);
        }
    }

    @Override
    public HttpRoute determineRoute(HttpHost host, HttpRequest request, HttpContext context)
            throws HttpException {
        if (host == null) {
            host = defaultHost;
        }
        return super.determineRoute(host, request, context);
    }

    @Override
    protected HttpHost determineProxy(HttpHost target, HttpRequest request,
            HttpContext context) {
        if (proxyHost != null // no proxy -> no need to check
                && (isTargetLocalhost(target) // first check localhost
                || (noProxyHostsEvaluator.shouldBypassProxy(target.toHostString())))) { // check host from list
            return null;
        }
        return proxyHost;
    }

    /**
     * @return The default host details to use in case the host is not part of the request
     */
    public HttpHost getDefaultHost() {
        return defaultHost;
    }

    /**
     * @return The proxy host details to use when executing requests
     */
    public HttpHost getProxy() { return proxyHost; }

    /**
     * @return String that contains list of hosts to bypass the proxy, as kept in
     * NoProxyHostsEvaluator Object
     */
    public String getNoProxyHosts() { return noProxyHostsEvaluator.getNoProxyHosts(); }

    private boolean isTargetLocalhost(HttpHost target) {
        return target.getHostName().equalsIgnoreCase("localhost")
                || target.getHostName().equalsIgnoreCase("127.0.0.1");
    }
}