package org.jfrog.client.util;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.Objects;

/**
 * @author Shay Bagants
 */
class InputStreamKeyStoreProvider implements KeyStoreProvider {
    private static final Logger log = LoggerFactory.getLogger(InputStreamKeyStoreProvider.class);

    private KeyStore keyStore;
    private char[] keyStorePasswordChar;

    InputStreamKeyStoreProvider(InputStream keyStoreInputStream, String keyStorePassword)
            throws KeyStoreProviderException {
        if (StringUtils.isBlank(keyStorePassword)) {
            throw new KeyStoreProviderException("Password cannot be empty");
        }
        if (keyStoreInputStream == null) {
            throw new KeyStoreProviderException("Stream is missing");
        }
        this.keyStorePasswordChar = keyStorePassword.toCharArray();
        try {
            KeyStore jks = KeyStore.getInstance("JKS");
            jks.load(keyStoreInputStream, keyStorePasswordChar);
            keyStore = jks;
        } catch (CertificateException | NoSuchAlgorithmException | KeyStoreException | IOException e) {
            log.error("Unable to load stream into keystore. {}", e.getMessage());
            log.debug("Unable to load stream into keystore.", e);
            throw new KeyStoreProviderException(e.getMessage(), e);
        } finally {
            try {
                if (keyStoreInputStream != null) {
                    keyStoreInputStream.close();
                }
            } catch (IOException ignored) {
            }
        }
    }

    @Override
    public KeyStore provide() throws KeyStoreException {
        return Objects.requireNonNull(keyStore);
    }

    @Override
    public char[] getPassword() {
        return keyStorePasswordChar;
    }
}
