package org.jfrog.common.config.diff;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.beanutils.ConversionException;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.Diff;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.apache.commons.lang3.reflect.TypeUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import javax.annotation.Nonnull;
import java.io.File;
import java.lang.reflect.*;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.jfrog.common.ExceptionUtils.wrapException;

/**
 * A utility class to merge list of diffs that were produced by the diff engine to an object
 *
 * @author Noam Shemesh
 */
public abstract class DiffMerger {
    static final String KEY_PLACEHOLDER_FORMAT = "{%d}";
    static final String KEY_PLACEHOLDER_FORMAT_REGEX = "\\{\\d+\\}";
    private static final Pattern KEY_PLACEHOLDER_REGEX =
            Pattern.compile("^([^" + DiffUtils.DELIMITER + "]*)" +
                    Pattern.quote(DiffUtils.DELIMITER + "{") + "(\\d+)" + Pattern.quote("}") + "[" + DiffUtils.DELIMITER + "]?(.*)$");
    private static final int BEFORE_KEY_PLACEHOLDER_GROUP = 1;
    private static final int KEY_PLACEHOLDER_GROUP = 2;
    private static final int AFTER_KEY_PLACEHOLDER_GROUP = 3;

    private DiffMerger() {}

    public static <T> T mergeDiffs(@Nonnull T object, Collection<DataDiff<?>> diffs) {
        Set<DataDiff> skipped = new HashSet<>(diffs);
        Map<String, Set<Object>> prev;

        Map<String, Set<Object>> refsDict = new HashMap<>();
        do {
            prev = refsDict;
            skipped = internalApplyDiffs(refsDict, object.getClass(), object, skipped);
            if (skipped.size() > 0) {
                refsDict = initReferencesDictIfNeeded(object);
            }
        } while (skipped.size() > 0 && !prev.keySet().containsAll(refsDict.keySet()));

        if (skipped.size() > 0) {
            throw new IllegalArgumentException("Reference rules weren't found. " + skipped);
        }

        return object;
    }

    private static <T> Map<String, Set<Object>> initReferencesDictIfNeeded(T object) {
        return FieldUtils.getFieldsListWithAnnotation(object.getClass(), DiffReferenceable.class).stream()
                .map(DiffUtils::fieldToMethod)
                .map(method -> wrapException(() -> Pair.of(method, method.invoke(object)), IllegalStateException.class))
                .flatMap(pair -> {
                    Object listOrMap = pair.getRight();
                    if (listOrMap instanceof Map) {
                        return (Stream<Pair<String, Object>>) ((Map<?, ?>) listOrMap).entrySet().stream() // Cast is a hint for maven compilation :(
                                .map(entry -> Pair.of(entry.getKey().toString(), (Object) entry.getValue()));
                    }
                    if (listOrMap instanceof Collection) {
                        Type[] genericTypes = ((ParameterizedType) pair.getLeft().getGenericReturnType()).getActualTypeArguments();
                        Method keyMethod = findKeyMethod(findParameterType(genericTypes[0]));

                        return ((List<?>) listOrMap).stream()
                                .map(entry -> wrapException(() -> Pair.of("" + keyMethod.invoke(entry), (Object) entry), IllegalStateException.class));
                    }

                    throw new IllegalStateException(DiffReferenceable.class.getSimpleName() + " must be over list or map return type");
                })
                .collect(Collectors.toMap(
                        Pair::getLeft, pair -> ImmutableSet.of(pair.getRight()),
                        DiffMerger::combineSets));
    }

    private static <T> Set<DataDiff> internalApplyDiffs(Map<String, Set<Object>> refsDict, Class<?> clazz, T object, Collection<DataDiff> diffs) {
        Class<?> impl = findImpl(clazz);

        Map<String, Set<DataDiff>> fieldNameToDiff = DiffUtils.diffForField(diffs, DataDiff::getFieldName);
        Map<String, Set<DataDiff>> firstLevel = compressFieldsToFirstLevel(fieldNameToDiff);

        Method[] allMethods = impl.getMethods();
        return firstLevel.entrySet().stream()
                .flatMap(entry -> {
                    if (assertValidKeyAndObject(object, entry)) {
                        return Stream.of();
                    }

                    Method getter = findGetter(allMethods, entry.getKey());

                    return processChild(refsDict, impl, object, entry.getKey(), entry.getValue(), allMethods, getter, DiffUtils.methodToField(getter))
                            .stream();
                })
                // Setting the key prefix again after removing it for the internal processing
                .map(newData -> new DataDiff<>(
                        (StringUtils.isEmpty(newData.getPrefixContext()) ? "" : newData.getPrefixContext() + DiffUtils.DELIMITER) +
                                newData.getFieldName(), newData.getNewValue()))
                .collect(Collectors.toSet());
    }

    private static <T> boolean assertValidKeyAndObject(T object, Map.Entry<String, Set<DataDiff>> entry) {
        if (StringUtils.isEmpty(entry.getKey())) {
            throw new IllegalArgumentException("Key must not be empty");
        }
        if (object == null) {
            throw new IllegalStateException("Object for key " + entry.getKey() + " is empty");
        }

        // Ignoring new object as they were handled while checking if current value is null (in processChild)
        return entry.getKey().equals(DiffUtils.NEW_MARKER);

    }

    private static Class<?> findImpl(Class<?> clazz) {
        if (clazz.getAnnotation(GenerateDiffFunction.class) == null) {
            throw new IllegalStateException("Class " + clazz.getName() + " must be annotated with " + GenerateDiffFunction.class.getSimpleName());
        }

        Class<?> impl = clazz;
        Class<?> defaultImpl = clazz.getAnnotation(GenerateDiffFunction.class).defaultImpl();
        if (ClassUtils.isAssignable(defaultImpl, clazz)) {
            impl = defaultImpl;
        }
        return impl;
    }

    private static <T> Set<DataDiff> processChild(Map<String, Set<Object>> refsDict, Class clazz, T object, String key, Set<DataDiff> value, Method[] allMethods,
            Method getter, Field field) {
        Class<?> param = getter.getReturnType();
        Set<DataDiff> referencedValue = value;
        boolean isReference = field.getAnnotation(DiffReference.class) != null;
        if (isReference) {
            referencedValue = parseReference(refsDict, value, getter.getGenericReturnType());
            if (referencedValue.isEmpty()) {
                return value;
            }
        }

        if (isReference || isPrimitiveOrWrapperOrAtomic(param, field) || isDeletedLeaf(referencedValue, key)) {
            Method setter = findSetter(allMethods, key);
            handlePrimitive(refsDict, referencedValue, diff -> invokeSetter(object, diff.getNewValue(), setter),
                    clazz.getSimpleName() + "#" + setter.getName(), setter.getParameterTypes()[0],
                    setter, isReference);
            return ImmutableSet.of();
        } else if (ClassUtils.isAssignable(param, Map.class) || ClassUtils.isAssignable(param, Collection.class)) {
            return handleMapOrCollection(refsDict, allMethods, object, referencedValue, getter, param, key);
        } else {
            String signature = object.getClass().getSimpleName() + "#" + getter.getName();
            Object childObject = wrapException(() -> getter.invoke(object), IllegalStateException.class, null, signature);
            if (childObject == null) {
                Method setter = findSetter(allMethods, key);
                Object newChildObject = initObjectForType(setter.getParameterTypes()[0], key);
                wrapException(() -> setter.invoke(object, newChildObject), IllegalStateException.class);
                childObject = wrapException(() -> getter.invoke(object), IllegalStateException.class, null, signature);
            }
            return internalApplyDiffs(refsDict, param, childObject, referencedValue.stream()
                    .map(diff -> new DataDiff<>(key, removeFieldPrefix(diff, key), diff.getOldValue(), diff.getNewValue()))
                    .collect(Collectors.toSet()));
        }
    }

    private static boolean isDeletedLeaf(Set<DataDiff> referencedValue, String key) {
        if (referencedValue.size() != 1) {
            return false;
        }
        DataDiff next = referencedValue.iterator().next();
        return next.getNewValue() == null && DiffUtils.DELETED_MARKER.equals(removeFieldPrefix(next, key));
    }

    private static Set<DataDiff> parseReference(Map<String, Set<Object>> refsDict, Set<DataDiff> value, Type param) {
        Set<DataDiff> referencedValue;
        referencedValue = value.stream()
                .map(data -> Pair.of(data, new DataDiff<>(data.getPrefixContext(), data.getFieldName(), data.getOldValue(),
                        refValueForObjectOrCollection(refsDict, data, param))))
                .filter(oldAndNewData -> // If old value is null meaning that it's a delete
                        oldAndNewData.getLeft().getNewValue() == null ||
                                (oldAndNewData.getRight().getNewValue() != null &&
                                        (!(oldAndNewData.getRight().getNewValue() instanceof Collection) ||
                                                !((Collection) oldAndNewData.getRight().getNewValue()).isEmpty())))
                .map(Pair::getRight)
                .collect(Collectors.toSet());
        return referencedValue;
    }

    private static Object refValueForObjectOrCollection(Map<String, Set<Object>> refsDict, DataDiff<Object> data,
            Type param) {
        if (data.getNewValue() instanceof Collection) {
            List<Object> list = ((Collection<?>) data.getNewValue()).stream()
                    .map(val -> refValueForObject(refsDict, val, findParameterType(((ParameterizedType) param).getActualTypeArguments()[0])))
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
            return data.getNewValue() instanceof Set ? new TreeSet<>(list) : list;
        }

        if (TypeUtils.isAssignable(param, Collection.class)) {
            return refValueForObject(refsDict, data.getNewValue(),
                    findParameterType(((ParameterizedType) param).getActualTypeArguments()[0]));
        }

        return refValueForObject(refsDict, data.getNewValue(), (Class) param);
    }

    private static Object refValueForObject(Map<String, Set<Object>> refsDict, Object data,
            Class<?> param) {
        return refsDict.getOrDefault(data, new HashSet<>()).stream()
                .filter(referenced -> ClassUtils.isAssignable(referenced.getClass(), param))
                .findFirst()
                .orElse(null);
    }

    private static Method findGetter(Method[] allMethods, String key) {
        String getterName = "get" + StringUtils.capitalize(key);
        String isserName = "is" + StringUtils.capitalize(key);

        return Stream.of(allMethods)
                .filter(method -> {
                    Field field;
                    try {
                        field = DiffUtils.methodToField(method);
                    } catch (IllegalStateException e) {
                        return false;
                    }
                    return getterDelegatedName(field, method, getterName, key) || getterDelegatedName(field, method, isserName, key);
                })
                .filter(method -> method.getParameterCount() == 0)
                .min(Comparator.comparingInt(DiffMerger::booleanToInt))
                .orElseThrow(() -> new IllegalArgumentException("Key \"" + key + "\" is not part of the configuration"));
    }

    private static int booleanToInt(Method first) {
        return first.getReturnType().isInterface() ? 1 : -1;
    }

    public static Method findSetter(Method[] allMethods, String key) {
        Method setter = optionallyFindSetter(allMethods, key)
                .orElseThrow(() -> new IllegalArgumentException("Key " + key + " is not changeable"));
        if (setter.getParameterCount() != 1) {
            throw new IllegalStateException("Parameter count for " + setter.getDeclaringClass().getName() + "#" +
                    setter.getName() + " must be 1");
        }

        return setter;
    }

    private static Optional<Method> optionallyFindSetter(Method[] allMethods, String key) {
        // Searching for the setter according to the getter to make sure we're consistent
        // (@DiffElement for example could change the getter that found)
        Method getter = findGetter(allMethods, key);
        String setterName = "set" + StringUtils.capitalize(DiffUtils.toFieldName(getter.getName()));

        return Stream.of(allMethods)
                .filter(method -> setterName.equals(method.getName()))
                .findFirst();
    }

    private static boolean getterDelegatedName(Field field, Method method, String methodExpected, String key) {
        DiffElement mergeAnnon = field.getAnnotation(DiffElement.class);
        return ((mergeAnnon != null && !StringUtils.isEmpty(mergeAnnon.name())) && key.equals(mergeAnnon.name()))
                || methodExpected.equals(method.getName());
    }

    public static boolean isPrimitiveOrWrapperOrAtomic(Class<?> param, Field field) {
        return ClassUtils.isPrimitiveOrWrapper(param) ||
                ClassUtils.isAssignable(param, String.class) ||
                ClassUtils.isAssignable(param, File.class) ||
                ClassUtils.isAssignable(param, Enum.class) ||
                (field != null && field.isAnnotationPresent(DiffAtomic.class)) ||
                (field != null && isCollectionOrMapOfPrimitive(field));
    }

    private static boolean isCollectionOrMapOfPrimitive(Field field) {
        if (!(field.getGenericType() instanceof ParameterizedType)) {
            return false;
        }

        Type[] types = ((ParameterizedType) field.getGenericType()).getActualTypeArguments();
        if (ClassUtils.isAssignable(field.getType(), Map.class) && types[1] instanceof Class) {
            return isPrimitiveOrWrapperOrAtomic((Class) types[1], null);
        }

        return ClassUtils.isAssignable(field.getType(), Collection.class) &&
                types[0] instanceof Class &&
                isPrimitiveOrWrapperOrAtomic((Class) types[0], null);

    }

    private static <T> Set<DataDiff> handleMapOrCollection(Map<String, Set<Object>> refsDict,
            Method[] allMethods, T mainObject, Set<DataDiff> value,
            Method getter, Class<?> param, String key) {
        Object childObject = wrapException(() -> getter.invoke(mainObject));
        Type[] genericTypes = ((ParameterizedType) getter.getGenericReturnType()).getActualTypeArguments();

        Class<?> parameterType;
        Map map;
        Collection collection;

        boolean isCollection = ClassUtils.isAssignable(param, Collection.class);
        if (isCollection) {
            parameterType = findParameterType(genericTypes[0]);
            Function<Object, String> keyMethodInvoker = invokeKeyMethod(parameterType);

            collection = (Collection<?>) childObject;
            map = ((Collection<?>) collection).stream()
                    .collect(Collectors.toMap(keyMethodInvoker, Function.identity()));
        } else {
            parameterType = findParameterType(genericTypes[1]);
            map = (Map) childObject;
            collection = null;
        }

        Set<DataDiff> toSkip = processMapOrCollectionElements(refsDict, key, value, parameterType, map, collection);

        optionallyFindSetter(allMethods, key).ifPresent(method -> invokeSetter(mainObject, isCollection ? collection : map, method));

        return toSkip;
    }

    public static Class<?> findParameterType(Type genericType) {
        if (genericType instanceof Class<?>) {
            return (Class<?>) genericType;
        }
        if (genericType instanceof WildcardType) {
            return findParameterType(((WildcardType) genericType).getUpperBounds()[0]);
        }

        throw new IllegalStateException("Unsupported type of generic: " + genericType);
    }

    public static Function<Object, String> invokeKeyMethod(Class<?> parameterType) {
        return object -> wrapException(() -> "" + findKeyMethod(parameterType).invoke(object), IllegalStateException.class);
    }

    public static Method findKeyMethod(Class<?> parameterType) {
        return DiffUtils.fieldToMethod(findKeyField(parameterType));

    }

    private static Field findKeyField(Class<?> parameterType) {
        return allFields(parameterType).stream()
                .filter(field -> field.getAnnotation(DiffKey.class) != null)
                .findFirst()
                .orElseThrow(() -> new IllegalStateException(
                        DiffKey.class.getSimpleName() + " is not present on any method on " + parameterType.getName()));
    }

    public static Set<Field> allFields(Class<?> clazz) {
        return ImmutableSet.<Field>builder()
                .addAll(Arrays.asList(clazz.getDeclaredFields()))
                .addAll(Object.class.equals(clazz.getSuperclass()) ? Collections.emptyList() : allFields(clazz.getSuperclass()))
                .build();
    }

    private static Set<DataDiff> processMapOrCollectionElements(Map<String, Set<Object>> refsDict,
            String key, Set<DataDiff> value, Class<?> parameterType, Map map, Collection collection) {
        Map<String, Set<DataDiff>> complexes = new HashMap<>();

        Set<DataDiff> skipped = replaceKeysPlaceholders(key, value, parameterType).stream()
                .flatMap(diff -> {
                    String[] splitted = diff.getFieldName().split("[" + DiffUtils.DELIMITER + "]");
                    if (splitted.length == 1) {
                        throw new IllegalArgumentException(
                                parameterType.getSimpleName() + " is a collection of data and " +
                                        "provided data is for single data point");
                    }

                    if (StringUtils.isEmpty(splitted[1])) {
                        throw new IllegalArgumentException("Key must not be empty");
                    }

                    if (processSpecialInstructionsForMap(map, collection, diff, splitted)) {
                        return Stream.empty();
                    }

                    if (ClassUtils.isPrimitiveOrWrapper(parameterType)) {
                        return handlePrimitive(refsDict, ImmutableSet.of(diff),
                                d -> handlePrimitiveForMapOrCollection(map, collection, diff, splitted[1]),
                                diff.getFieldName(), parameterType,
                                null, false)
                                .stream();
                    }
                    String prefix = splitted[0] + DiffUtils.DELIMITER + splitted[1];

                    String keyField = DiffUtils.toFieldName(findKeyMethod(parameterType).getName());
                    DataDiff<Object> keyNewData = new DataDiff<>(prefix, keyField, null, splitted[1]);

                    complexes.computeIfAbsent(splitted[1], x -> new HashSet<>())
                            .add(new DataDiff<>(prefix, removeFieldPrefix(diff, prefix), diff.getOldValue(), diff.getNewValue()));
                    complexes.get(splitted[1]).add(keyNewData);

                    return Stream.empty();
                }).collect(Collectors.toSet());

        if (!skipped.isEmpty()) {
            return skipped;
        }

        return complexes.entrySet().stream()
                .peek(entry -> putIfAbsent(parameterType, map, collection, entry.getKey()))
                .flatMap(entry -> ((Set<DataDiff>) internalApplyDiffs(refsDict,
                        (Class) parameterType, map.get(entry.getKey()), entry.getValue())).stream())
                .collect(Collectors.toSet());
    }

    private static void putIfAbsent(Class<?> parameterType, Map map, Collection collection, String key) {
        if (!map.containsKey(key)) {
            Object newInstance = initObjectForType(parameterType, key);

            if (collection != null) {
                collection.add(newInstance);
            }

            map.put(key, newInstance);
        }
    }

    private static Object initObjectForType(Class<?> parameterType, String key) {
        Class<?> impl = findImpl(parameterType);
        Object newInstance;
        try {
            newInstance = impl.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            throw new IllegalArgumentException("Could not create " + impl.getSimpleName() + " for key " + key);
        }
        return newInstance;
    }

    private static Set<DataDiff> replaceKeysPlaceholders(String key, Set<DataDiff> value,
            Class<?> parameterType) {
        String keyPropertyName = DiffUtils.toFieldName(findKeyMethod(parameterType).getName());

        Map<String, Object> keyDict = value.stream()
                .map(elem -> Pair.of(elem, KEY_PLACEHOLDER_REGEX.matcher(elem.getFieldName())))
                .filter(pair -> matches(pair.getRight(), key))
                .filter(pair -> pair.getRight().group(AFTER_KEY_PLACEHOLDER_GROUP).equals(keyPropertyName))
                .map(pair -> Pair.of(pair.getLeft(), pair.getRight().group(KEY_PLACEHOLDER_GROUP)))
                .collect(Collectors.toMap(Pair::getRight, pair -> pair.getLeft().getNewValue()));

        if (keyDict.values().stream().distinct().count() < keyDict.size()) {
            throw new IllegalArgumentException("Multiple not distinct keys for " + key);
        }

        return value.stream()
                .map(elem -> {
                    Matcher matcher = KEY_PLACEHOLDER_REGEX.matcher(elem.getFieldName());
                    if (matches(matcher, key)) {
                        return new DataDiff<>(
                                matcher.group(BEFORE_KEY_PLACEHOLDER_GROUP) + DiffUtils.DELIMITER +
                                        matcher.group(KEY_PLACEHOLDER_GROUP),
                                matcher.group(BEFORE_KEY_PLACEHOLDER_GROUP) + DiffUtils.DELIMITER +
                                        keyDict.get(matcher.group(KEY_PLACEHOLDER_GROUP)) + DiffUtils.DELIMITER +
                                        matcher.group(AFTER_KEY_PLACEHOLDER_GROUP),
                                elem.getOldValue(), elem.getNewValue());
                    }

                    return elem;
                })
                .collect(Collectors.toSet());

    }

    private static boolean matches(Matcher matcher, String key) {
        return matcher.matches() && matcher.group(BEFORE_KEY_PLACEHOLDER_GROUP).equals(key);
    }

    private static void handlePrimitiveForMapOrCollection(Map map, Collection collection,
            DataDiff diff, String key) {
        if (collection != null) {
            if (collection instanceof List) {
                List list = (List) collection;
                int indexOf = list.indexOf(map.get(key));
                if (indexOf < 0) {
                    list.add(diff.getNewValue());
                    return;
                }

                list.set(list.indexOf(map.get(key)), diff.getNewValue());
                return;
            }

            throw new IllegalStateException("Auto merge of " + collection.getClass().getSimpleName() + " is not supported");
        }

        map.put(key, diff.getNewValue());
    }

    private static boolean processSpecialInstructionsForMap(Map map, Collection collection, DataDiff diff,
            String[] splitted) {
        if (splitted.length == 3) {
            if (splitted[2].equals(DiffUtils.NEW_MARKER)) {
                if (collection != null) {
                    if (map.containsKey(splitted[1])) {
                        collection.remove(map.get(splitted[1]));
                    }
                    collection.add(diff.getNewValue());
                } else {
                    map.put(splitted[1], diff.getNewValue());
                }
                return true;
            }

            if (splitted[2].equals(DiffUtils.DELETED_MARKER)) {
                if (collection != null) {
                    collection.remove(map.get(splitted[1]));
                } else {
                    map.remove(splitted[1]);
                }
                return true;
            }
        }

        return false;
    }

    private static Set<DataDiff> handlePrimitive(
            Map<String, Set<Object>> refsDict, Set<DataDiff> value, Consumer<DataDiff<?>> invoke,
            String signature, Class<?> type, Method setter, boolean isReferenced) {

        if (setter != null && ClassUtils.isAssignable(type, Collection.class) &&
                // list->[value-name]->value
                (value.stream().allMatch(data -> StringUtils.countMatches(data.getFieldName(), DiffUtils.DELIMITER) >= 2) ||
                        value.stream().allMatch(data -> KEY_PLACEHOLDER_REGEX.matcher(data.getFieldName()).matches()))) {
            try {
                Type collectionType =
                        ((ParameterizedType) setter.getGenericParameterTypes()[0]).getActualTypeArguments()[0];

                Set<DataDiff> toProcess = isReferenced ? value :
                        newInstanceForEntities(refsDict, value, signature, findParameterType(collectionType));
                extractPrimitiveCollection(type, toProcess, signature)
                        .ifPresent(invoke);
            } catch (SkippedException e) {
                return e.data;
            }
            return Collections.emptySet();
        }

        if (value.size() != 1) {
            throw new IllegalArgumentException("Number of diffs count for primitive must be 1 [" + signature + "] " + value);
        }

        value.stream().map(v -> convertPrimitive(type, v)).forEach(invoke);

        return Collections.emptySet();
    }

    public static List<DataDiff<?>> diffToDataDiff(List<Diff<?>> diffs) {
        return diffs.stream()
                .map(currDiff -> new DataDiff<>(currDiff.getFieldName(), currDiff.getLeft(), currDiff.getRight()))
                .collect(Collectors.toList());
    }

    /**
     * Believe me, every other way is worse
     */
    private static class SkippedException extends RuntimeException {
        final Set<DataDiff> data;

        SkippedException(Set<DataDiff> data) {
            this.data = data;
        }
    }

    private static Set<DataDiff> newInstanceForEntities(
            Map<String, Set<Object>> refsDict, Set<DataDiff> value, String signature, Class<?> entitiesType) {
        if (isPrimitiveOrWrapperOrAtomic(entitiesType, null)) {
            return value.stream().map(
                    diff -> new DataDiff<>(diff.getPrefixContext(), diff.getFieldName(), diff.getOldValue(),
                            findValueToSet(diff.getNewValue(), entitiesType)))
                    .collect(Collectors.toSet());
        }

        Map<String, Set<DataDiff>> data = value.stream()
                .map(DiffMerger::fieldKeyToEntity)
                .collect(Collectors.toMap(Pair::getLeft, pair ->
                                ImmutableSet.<DataDiff>of(pair.getRight()),
                        DiffMerger::combineSets));

        return data.entrySet().stream()
                .map(perEntity -> newObjectForEntity(refsDict, perEntity, entitiesType, signature))
                .collect(Collectors.toSet());
    }

    private static Pair<String, DataDiff<Object>> fieldKeyToEntity(DataDiff entity) {
        Matcher matcher = KEY_PLACEHOLDER_REGEX.matcher(entity.getFieldName());

        String key;
        if (matcher.matches()) {
            key = matcher.group(BEFORE_KEY_PLACEHOLDER_GROUP) +
                    DiffUtils.DELIMITER + String.format(KEY_PLACEHOLDER_FORMAT,
                    Integer.parseInt(matcher.group(KEY_PLACEHOLDER_GROUP)));
        } else {
            String[] split = entity.getFieldName().split(DiffUtils.DELIMITER, 3);
            if (split.length != 3) {
                throw new IllegalStateException("Unexpected field name format " + entity.getFieldName());
            }

            key = split[0] + DiffUtils.DELIMITER + split[1];
        }
        return Pair.of(key, new DataDiff<>(key, removeFieldPrefix(entity, key), entity.getOldValue(), entity.getNewValue()));
    }

    private static DataDiff<Object> newObjectForEntity(
            Map<String, Set<Object>> refsDict,
            Map.Entry<String, Set<DataDiff>> perEntity,
            Class<?> entitiesType, String signature) {
        DataDiff first = perEntity.getValue().iterator().next();
        Object newObject = initObjectForType(entitiesType, signature);
        String[] splitted = first.getPrefixContext().split(DiffUtils.DELIMITER);
        if (splitted.length == 2 && !splitted[1].matches(KEY_PLACEHOLDER_FORMAT_REGEX)) {
            invokeSetter(newObject, splitted[1],
                    findSetter(entitiesType.getMethods(), findKeyField(entitiesType).getName()));
        }

        Set<DataDiff> skipped = internalApplyDiffs(refsDict, entitiesType,
                newObject, perEntity.getValue());

        if (!skipped.isEmpty()) {
            throw new SkippedException(skipped);
        }

        return new DataDiff<>(first.getPrefixContext(), perEntity.getKey(), null, newObject);
    }

    private static DataDiff<?> convertPrimitive(Class<?> type, DataDiff<?> v) {
        if (ClassUtils.isPrimitiveOrWrapper(type)) {
            try {
                if (v.getNewValue() == null && ClassUtils.isPrimitiveWrapper(type)) {
                    return v;
                }

                return new DataDiff<>(v.getPrefixContext(), v.getFieldName(),
                        v.getOldValue(), ConvertUtils.convert(v.getNewValue(), type));
            } catch (ConversionException e) {
                throw new IllegalArgumentException(e.getMessage(), e);
            }
        }
        return v;
    }

    private static Optional<DataDiff<Collection>> extractPrimitiveCollection(Class<?> type,
            Set<DataDiff> value, String signature) {
        List<Pair<String, Object>> data;
        if (value.stream()
                .allMatch(prop -> KEY_PLACEHOLDER_REGEX.matcher(prop.getFieldName()).matches())) {
            data = extractKeyPlaceholderFromRegex(value, signature);
        } else {
            data = extractKeyPlaceholderFromConcrete(value);
        }

        if (!data.isEmpty()) {
            String fieldName = data.get(0).getLeft();
            Collection<Object> primitiveCollection = data.stream()
                    .peek(entity -> {
                        if (!fieldName.equals(entity.getLeft())) {
                            throw new IllegalStateException("Illegal data for " + signature + ". data: " + value);
                        }
                    })
                    .map(Pair::getRight)
                    .collect(Collectors.toList());

            if (ClassUtils.isAssignable(type, Set.class)) {
                primitiveCollection = new TreeSet<>(primitiveCollection);
            }

            return Optional.of(new DataDiff<>(fieldName, primitiveCollection));
        }
        return Optional.empty();
    }

    private static List<Pair<String, Object>> extractKeyPlaceholderFromConcrete(Set<DataDiff> value) {
        return value.stream()
                .map(entity -> Pair.of(entity.getFieldName(), entity.getNewValue()))
                .collect(Collectors.toList());
    }

    private static List<Pair<String, Object>> extractKeyPlaceholderFromRegex(Set<DataDiff> value, String signature) {
        return value.stream()
                .map(entity -> {
                    Matcher matcher = KEY_PLACEHOLDER_REGEX.matcher(entity.getFieldName());
                    if (matcher.matches() && matcher.group(AFTER_KEY_PLACEHOLDER_GROUP).isEmpty()) {
                        return Triple.of(matcher.group(BEFORE_KEY_PLACEHOLDER_GROUP),
                                Integer.parseInt(matcher.group(KEY_PLACEHOLDER_GROUP)), entity.getNewValue());
                    }

                    throw new IllegalStateException(signature + " must be in format fieldname.{position}");
                })
                .sequential()
                .sorted(Comparator.comparingInt(Triple::getMiddle))
                .map(triple -> Pair.of(triple.getLeft(), triple.getRight()))
                .collect(Collectors.toList());
    }

    private static <T> void invokeSetter(T object, Object value, Method setter) {
        try {
            Object valueToSet = findValueToSet(value, setter.getParameterTypes()[0]);
            setter.invoke(object, valueToSet);
        } catch (IllegalAccessException | InvocationTargetException e) {
            throw new IllegalArgumentException("Cannot invoke " +
                    setter.getDeclaringClass().getName() + "#" + setter.getName(), e);
        }
    }

    private static Object findValueToSet(Object value, Class<?> param) {
        if (param.isEnum() &&
                ClassUtils.isAssignable(value.getClass(), String.class)) {
            return findEnum(param, (String) value);
        } else {
            return value;
        }
    }

    private static Object findEnum(Class<?> enumClass, String name) {
        GenerateDiffFunction annotation = enumClass.getAnnotation(GenerateDiffFunction.class);
        if (annotation == null) {
            throw new IllegalStateException("Enum class is not implementing " + GenerateDiffFunction.class.getSimpleName());
        }
        Method typeMethod;
        try {
            typeMethod = enumClass.getMethod(annotation.typeMethod());
        } catch (NoSuchMethodException e) {
            throw new IllegalStateException("Method " + annotation.typeMethod() +
                    " that was referred as typeMethod was not found in " + enumClass.getSimpleName());
        }

        return Stream.of(enumClass.getEnumConstants())
                .filter(enValue -> wrapException(() -> typeMethod.invoke(enValue)).equals(name))
                .findFirst()
                .orElseThrow(() ->
                        new IllegalArgumentException(enumClass.getSimpleName() + ": Enum constant wasn't found for " + name));
    }

    private static Map<String, Set<DataDiff>> compressFieldsToFirstLevel(Map<String, Set<DataDiff>> allDiffs) {
        return allDiffs.entrySet().stream()
                .collect(Collectors.toMap(
                        entry -> findParentKey(entry.getKey()),
                        entry -> ImmutableSet.copyOf(entry.getValue()),
                        DiffMerger::combineSets));
    }

    private static <U> Set<U> combineSets(Set<U> set1, Set<U> set2) {
        return ImmutableSet.<U>builder().addAll(set1).addAll(set2).build();
    }

    private static String findParentKey(String key) {
        if (!key.contains(DiffUtils.DELIMITER)) {
            return key;
        }
        return key.substring(0, key.indexOf(DiffUtils.DELIMITER));
    }

    private static String removeFieldPrefix(DataDiff other, String prefix) {
        return other.getFieldName().replaceFirst("^" + Pattern.quote(prefix + DiffUtils.DELIMITER), "");
    }
}
