/*
 * Decompiled with CFR 0.152.
 */
package org.jetbrains.kotlin.com.intellij.openapi.util.io;

import com.sun.jna.Library;
import com.sun.jna.Memory;
import com.sun.jna.Native;
import com.sun.jna.Pointer;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.kotlin.com.intellij.openapi.diagnostic.Logger;
import org.jetbrains.kotlin.com.intellij.openapi.util.SystemInfo;
import org.jetbrains.kotlin.com.intellij.openapi.util.io.FileAttributes;
import org.jetbrains.kotlin.com.intellij.openapi.util.io.win32.FileInfo;
import org.jetbrains.kotlin.com.intellij.openapi.util.io.win32.IdeaWin32;
import org.jetbrains.kotlin.com.intellij.util.ArrayUtil;
import org.jetbrains.kotlin.com.intellij.util.BitUtil;
import org.jetbrains.kotlin.com.intellij.util.SystemProperties;
import org.jetbrains.kotlin.com.intellij.util.containers.ContainerUtil;

public class FileSystemUtil {
    private static final Logger LOG = Logger.getInstance("#com.intellij.openapi.util.io.FileSystemUtil");
    @NotNull
    private static Mediator ourMediator = FileSystemUtil.getMediator();

    private static Mediator getMediator() {
        boolean forceNio2 = SystemProperties.getBooleanProperty("idea.io.use.nio2", false);
        boolean forceFallback = SystemProperties.getBooleanProperty("idea.io.use.fallback", false);
        Throwable error = null;
        if (!forceNio2 && !forceFallback) {
            if (SystemInfo.isWindows && IdeaWin32.isAvailable()) {
                try {
                    return FileSystemUtil.check(new IdeaWin32MediatorImpl());
                }
                catch (Throwable t) {
                    error = t;
                }
            } else if (SystemInfo.isLinux || SystemInfo.isMac || SystemInfo.isSolaris || SystemInfo.isFreeBSD) {
                try {
                    return FileSystemUtil.check(new JnaUnixMediatorImpl());
                }
                catch (Throwable t) {
                    error = t;
                }
            }
        }
        if (!forceFallback && SystemInfo.isJavaVersionAtLeast("1.7") && !"1.7.0-ea".equals(SystemInfo.JAVA_VERSION)) {
            try {
                return FileSystemUtil.check(new Nio2MediatorImpl());
            }
            catch (Throwable t) {
                error = t;
            }
        }
        if (!forceFallback) {
            LOG.warn("Failed to load filesystem access layer: " + SystemInfo.OS_NAME + ", " + SystemInfo.JAVA_VERSION + ", " + "nio2=" + forceNio2, error);
        }
        return new FallbackMediatorImpl();
    }

    private static Mediator check(Mediator mediator) throws Exception {
        String quickTestPath = SystemInfo.isWindows ? "C:\\" : "/";
        mediator.getAttributes(quickTestPath);
        return mediator;
    }

    @Nullable
    public static FileAttributes getAttributes(@NotNull String path) {
        if (path == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil", "getAttributes"));
        }
        try {
            return ourMediator.getAttributes(path);
        }
        catch (Exception e) {
            LOG.warn(e);
            return null;
        }
    }

    @Nullable
    public static FileAttributes getAttributes(@NotNull File file) {
        if (file == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "file", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil", "getAttributes"));
        }
        return FileSystemUtil.getAttributes(file.getPath());
    }

    public static boolean isSymLink(@NotNull String path) {
        if (path == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil", "isSymLink"));
        }
        if (SystemInfo.areSymLinksSupported) {
            FileAttributes attributes = FileSystemUtil.getAttributes(path);
            return attributes != null && attributes.isSymLink();
        }
        return false;
    }

    public static boolean isSymLink(@NotNull File file) {
        if (file == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "file", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil", "isSymLink"));
        }
        return FileSystemUtil.isSymLink(file.getAbsolutePath());
    }

    public static boolean clonePermissionsToExecute(@NotNull String source, @NotNull String target) {
        if (source == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil", "clonePermissionsToExecute"));
        }
        if (target == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil", "clonePermissionsToExecute"));
        }
        try {
            return ourMediator.clonePermissions(source, target, true);
        }
        catch (Exception e) {
            LOG.warn(e);
            return false;
        }
    }

    private static class FallbackMediatorImpl
    extends Mediator {
        private final Object myFileSystem;
        private final Method myGetBooleanAttributes;

        private FallbackMediatorImpl() {
            Method getBooleanAttributes;
            Object fileSystem;
            try {
                Field fs = File.class.getDeclaredField("fs");
                fs.setAccessible(true);
                fileSystem = fs.get(null);
                getBooleanAttributes = fileSystem.getClass().getMethod("getBooleanAttributes", File.class);
                getBooleanAttributes.setAccessible(true);
            }
            catch (Throwable t) {
                fileSystem = null;
                getBooleanAttributes = null;
            }
            this.myFileSystem = fileSystem;
            this.myGetBooleanAttributes = getBooleanAttributes;
        }

        @Override
        protected FileAttributes getAttributes(@NotNull String path) throws Exception {
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$FallbackMediatorImpl", "getAttributes"));
            }
            File file = new File(path);
            if (this.myFileSystem != null) {
                int flags = (Integer)this.myGetBooleanAttributes.invoke(this.myFileSystem, file);
                if (flags != 0) {
                    boolean isDirectory = BitUtil.isSet(flags, 4);
                    boolean isSpecial = BitUtil.notSet(flags, 2) && BitUtil.notSet(flags, 4);
                    boolean isHidden = BitUtil.isSet(flags, 8) && !FallbackMediatorImpl.isWindowsRoot(path);
                    return new FileAttributes(isDirectory, isSpecial, false, isHidden, file.length(), file.lastModified(), file.canWrite());
                }
            } else if (file.exists()) {
                boolean isDirectory = file.isDirectory();
                boolean isSpecial = !isDirectory && !file.isFile();
                boolean isHidden = file.isHidden() && !FallbackMediatorImpl.isWindowsRoot(path);
                return new FileAttributes(isDirectory, isSpecial, false, isHidden, file.length(), file.lastModified(), file.canWrite());
            }
            return null;
        }

        private static boolean isWindowsRoot(String p) {
            return SystemInfo.isWindows && p.length() >= 2 && p.length() <= 3 && Character.isLetter(p.charAt(0)) && p.charAt(1) == ':';
        }

        @Override
        protected boolean clonePermissions(@NotNull String source, @NotNull String target, boolean onlyPermissionsToExecute) throws Exception {
            if (source == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$FallbackMediatorImpl", "clonePermissions"));
            }
            if (target == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$FallbackMediatorImpl", "clonePermissions"));
            }
            if (SystemInfo.isUnix) {
                File srcFile = new File(source);
                File dstFile = new File(target);
                if (!onlyPermissionsToExecute && !dstFile.setWritable(srcFile.canWrite(), true)) {
                    return false;
                }
                return dstFile.setExecutable(srcFile.canExecute(), true);
            }
            return false;
        }
    }

    private static class JnaUnixMediatorImpl
    extends Mediator {
        private static final int[] LINUX_32 = new int[]{16, 44, 72, 24, 28};
        private static final int[] LINUX_64 = new int[]{24, 48, 88, 28, 32};
        private static final int[] LNX_PPC32 = new int[]{16, 48, 80, 24, 28};
        private static final int[] LNX_PPC64 = LINUX_64;
        private static final int[] LINUX_ARM = LNX_PPC32;
        private static final int[] BSD_32 = new int[]{8, 48, 32, 12, 16};
        private static final int[] BSD_64 = new int[]{8, 72, 40, 12, 16};
        private static final int[] SUN_OS_32 = new int[]{20, 48, 64, 28, 32};
        private static final int[] SUN_OS_64 = new int[]{16, 40, 64, 24, 28};
        private final LibC myLibC;
        private final int[] myOffsets;
        private final int myUid;
        private final int myGid;
        private final boolean myCoarseTs = SystemProperties.getBooleanProperty("idea.io.coarse.ts", false);

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        private JnaUnixMediatorImpl() throws Exception {
            if (SystemInfo.isLinux) {
                if ("arm".equals(SystemInfo.OS_ARCH)) {
                    if (!SystemInfo.is32Bit) throw new IllegalStateException("AArch64 architecture is not supported");
                    this.myOffsets = LINUX_ARM;
                } else {
                    this.myOffsets = "ppc".equals(SystemInfo.OS_ARCH) ? (SystemInfo.is32Bit ? LNX_PPC32 : LNX_PPC64) : (SystemInfo.is32Bit ? LINUX_32 : LINUX_64);
                }
            } else if (SystemInfo.isMac | SystemInfo.isFreeBSD) {
                this.myOffsets = SystemInfo.is32Bit ? BSD_32 : BSD_64;
            } else {
                if (!SystemInfo.isSolaris) throw new IllegalStateException("Unsupported OS/arch: " + SystemInfo.OS_NAME + "/" + SystemInfo.OS_ARCH);
                this.myOffsets = SystemInfo.is32Bit ? SUN_OS_32 : SUN_OS_64;
            }
            this.myLibC = (LibC)Native.loadLibrary("c", LibC.class);
            this.myUid = this.myLibC.getuid();
            this.myGid = this.myLibC.getgid();
        }

        @Override
        protected FileAttributes getAttributes(@NotNull String path) throws Exception {
            long mTime1;
            boolean isSymlink;
            int res;
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$JnaUnixMediatorImpl", "getAttributes"));
            }
            Memory buffer = new Memory(256L);
            int n = res = SystemInfo.isLinux ? this.myLibC.__lxstat64(1, path, buffer) : this.myLibC.lstat(path, buffer);
            if (res != 0) {
                return null;
            }
            int mode = this.getModeFlags(buffer) & 0xFFFF;
            boolean bl = isSymlink = (mode & 0xF000) == 40960;
            if (isSymlink) {
                if (!this.loadFileStatus(path, buffer)) {
                    return FileAttributes.BROKEN_SYMLINK;
                }
                mode = this.getModeFlags(buffer) & 0xFFFF;
            }
            boolean isDirectory = (mode & 0xF000) == 16384;
            boolean isSpecial = !isDirectory && (mode & 0xF000) != 32768;
            long size = buffer.getLong(this.myOffsets[1]);
            long l = mTime1 = SystemInfo.is32Bit ? (long)buffer.getInt(this.myOffsets[2]) : buffer.getLong(this.myOffsets[2]);
            long mTime2 = this.myCoarseTs ? 0L : (SystemInfo.is32Bit ? (long)buffer.getInt(this.myOffsets[2] + 4) : buffer.getLong(this.myOffsets[2] + 8));
            long mTime = mTime1 * 1000L + mTime2 / 1000000L;
            boolean writable = this.ownFile(buffer) ? (mode & 0x92) != 0 : this.myLibC.access(path, 2) == 0;
            return new FileAttributes(isDirectory, isSpecial, isSymlink, false, size, mTime, writable);
        }

        private boolean loadFileStatus(@NotNull String path, Memory buffer) {
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$JnaUnixMediatorImpl", "loadFileStatus"));
            }
            return (SystemInfo.isLinux ? this.myLibC.__xstat64(1, path, buffer) : this.myLibC.stat(path, buffer)) == 0;
        }

        @Override
        protected boolean clonePermissions(@NotNull String source, @NotNull String target, boolean onlyPermissionsToExecute) throws Exception {
            int permissions;
            if (source == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$JnaUnixMediatorImpl", "clonePermissions"));
            }
            if (target == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$JnaUnixMediatorImpl", "clonePermissions"));
            }
            Memory buffer = new Memory(256L);
            if (!this.loadFileStatus(source, buffer)) {
                return false;
            }
            int sourcePermissions = this.getModeFlags(buffer) & 0x1FF;
            if (onlyPermissionsToExecute) {
                if (!this.loadFileStatus(target, buffer)) {
                    return false;
                }
                int targetPermissions = this.getModeFlags(buffer) & 0x1FF;
                permissions = targetPermissions & 0xFFFFFFB6 | sourcePermissions & 0x49;
            } else {
                permissions = sourcePermissions;
            }
            return this.myLibC.chmod(target, permissions) == 0;
        }

        private int getModeFlags(Memory buffer) {
            return SystemInfo.isLinux ? buffer.getInt(this.myOffsets[0]) : (int)buffer.getShort(this.myOffsets[0]);
        }

        private boolean ownFile(Memory buffer) {
            return buffer.getInt(this.myOffsets[3]) == this.myUid && buffer.getInt(this.myOffsets[4]) == this.myGid;
        }

        private static interface LibC
        extends Library {
            public int getuid();

            public int getgid();

            public int lstat(String var1, Pointer var2);

            public int stat(String var1, Pointer var2);

            public int __lxstat64(int var1, String var2, Pointer var3);

            public int __xstat64(int var1, String var2, Pointer var3);

            public int chmod(String var1, int var2);

            public int access(String var1, int var2);
        }
    }

    private static class IdeaWin32MediatorImpl
    extends Mediator {
        private IdeaWin32 myInstance = IdeaWin32.getInstance();

        private IdeaWin32MediatorImpl() {
        }

        @Override
        protected FileAttributes getAttributes(@NotNull String path) throws Exception {
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$IdeaWin32MediatorImpl", "getAttributes"));
            }
            FileInfo fileInfo = this.myInstance.getInfo(path);
            return fileInfo != null ? fileInfo.toFileAttributes() : null;
        }
    }

    private static class Nio2MediatorImpl
    extends Mediator {
        private final Object myDefaultFileSystem = Class.forName("java.nio.file.FileSystems").getMethod("getDefault", new Class[0]).invoke(null, new Object[0]);
        private final Method myGetPath;
        private final Method myIsSymbolicLink;
        private final Object myLinkOptions;
        private final Object myNoFollowLinkOptions;
        private final Method myReadAttributes;
        private final Method mySetAttribute;
        private final Method myToMillis;
        private final String mySchema;

        private Nio2MediatorImpl() throws Exception {
            Class<?> fsClass = Class.forName("java.nio.file.FileSystem");
            this.myGetPath = fsClass.getMethod("getPath", String.class, String[].class);
            this.myGetPath.setAccessible(true);
            Class<?> pathClass = Class.forName("java.nio.file.Path");
            Class<?> filesClass = Class.forName("java.nio.file.Files");
            this.myIsSymbolicLink = filesClass.getMethod("isSymbolicLink", pathClass);
            this.myIsSymbolicLink.setAccessible(true);
            Class<?> linkOptClass = Class.forName("java.nio.file.LinkOption");
            this.myLinkOptions = Array.newInstance(linkOptClass, 0);
            this.myNoFollowLinkOptions = Array.newInstance(linkOptClass, 1);
            Array.set(this.myNoFollowLinkOptions, 0, linkOptClass.getField("NOFOLLOW_LINKS").get(null));
            Class<?> linkOptArrClass = this.myLinkOptions.getClass();
            this.myReadAttributes = filesClass.getMethod("readAttributes", pathClass, String.class, linkOptArrClass);
            this.myReadAttributes.setAccessible(true);
            this.mySetAttribute = filesClass.getMethod("setAttribute", pathClass, String.class, Object.class, linkOptArrClass);
            this.mySetAttribute.setAccessible(true);
            Class<?> fileTimeClass = Class.forName("java.nio.file.attribute.FileTime");
            this.myToMillis = fileTimeClass.getMethod("toMillis", new Class[0]);
            this.myToMillis.setAccessible(true);
            this.mySchema = SystemInfo.isWindows ? "dos:*" : "posix:*";
        }

        @Override
        protected FileAttributes getAttributes(@NotNull String path) throws Exception {
            if (path == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$Nio2MediatorImpl", "getAttributes"));
            }
            try {
                boolean isSymbolicLink;
                Map attributes;
                block8: {
                    Object pathObj = this.myGetPath.invoke(this.myDefaultFileSystem, path, ArrayUtil.EMPTY_STRING_ARRAY);
                    attributes = (Map)this.myReadAttributes.invoke(null, pathObj, this.mySchema, this.myNoFollowLinkOptions);
                    isSymbolicLink = (Boolean)attributes.get("isSymbolicLink");
                    if (isSymbolicLink) {
                        try {
                            attributes = (Map)this.myReadAttributes.invoke(null, pathObj, this.mySchema, this.myLinkOptions);
                        }
                        catch (InvocationTargetException e) {
                            Throwable cause = e.getCause();
                            if (cause == null || !"java.nio.file.NoSuchFileException".equals(cause.getClass().getName())) break block8;
                            return FileAttributes.BROKEN_SYMLINK;
                        }
                    }
                }
                boolean isDirectory = (Boolean)attributes.get("isDirectory");
                boolean isOther = (Boolean)attributes.get("isOther");
                long size = (Long)attributes.get("size");
                long lastModified = (Long)this.myToMillis.invoke(attributes.get("lastModifiedTime"), new Object[0]);
                if (SystemInfo.isWindows) {
                    boolean isHidden = new File(path).getParent() == null ? false : (Boolean)attributes.get("hidden");
                    boolean isWritable = (Boolean)attributes.get("readonly") == false;
                    return new FileAttributes(isDirectory, isOther, isSymbolicLink, isHidden, size, lastModified, isWritable);
                }
                boolean isWritable = new File(path).canWrite();
                return new FileAttributes(isDirectory, isOther, isSymbolicLink, false, size, lastModified, isWritable);
            }
            catch (InvocationTargetException e) {
                Throwable cause = e.getCause();
                if (cause instanceof IOException || cause != null && "java.nio.file.InvalidPathException".equals(cause.getClass().getName())) {
                    LOG.debug(cause);
                    return null;
                }
                throw e;
            }
        }

        @Override
        protected boolean clonePermissions(@NotNull String source, @NotNull String target, boolean onlyPermissionsToExecute) throws Exception {
            if (source == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$Nio2MediatorImpl", "clonePermissions"));
            }
            if (target == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$Nio2MediatorImpl", "clonePermissions"));
            }
            if (SystemInfo.isUnix) {
                Object sourcePath = this.myGetPath.invoke(this.myDefaultFileSystem, source, ArrayUtil.EMPTY_STRING_ARRAY);
                Object targetPath = this.myGetPath.invoke(this.myDefaultFileSystem, target, ArrayUtil.EMPTY_STRING_ARRAY);
                Collection sourcePermissions = this.getPermissions(sourcePath);
                Collection targetPermissions = this.getPermissions(targetPath);
                if (sourcePermissions != null && targetPermissions != null) {
                    if (onlyPermissionsToExecute) {
                        HashSet permissionsToSet = ContainerUtil.newHashSet();
                        for (Object permission : targetPermissions) {
                            if (permission.toString().endsWith("_EXECUTE")) continue;
                            permissionsToSet.add(permission);
                        }
                        for (Object permission : sourcePermissions) {
                            if (!permission.toString().endsWith("_EXECUTE")) continue;
                            permissionsToSet.add(permission);
                        }
                        this.mySetAttribute.invoke(null, targetPath, "posix:permissions", permissionsToSet, this.myLinkOptions);
                    } else {
                        this.mySetAttribute.invoke(null, targetPath, "posix:permissions", sourcePermissions, this.myLinkOptions);
                    }
                    return true;
                }
            }
            return false;
        }

        private Collection getPermissions(Object sourcePath) throws IllegalAccessException, InvocationTargetException {
            Map attributes = (Map)this.myReadAttributes.invoke(null, sourcePath, "posix:permissions", this.myLinkOptions);
            if (attributes == null) {
                return null;
            }
            Object permissions = attributes.get("permissions");
            return permissions instanceof Collection ? (Collection)permissions : null;
        }
    }

    private static abstract class Mediator {
        private Mediator() {
        }

        @Nullable
        protected abstract FileAttributes getAttributes(@NotNull String var1) throws Exception;

        protected boolean clonePermissions(@NotNull String source, @NotNull String target, boolean onlyPermissionsToExecute) throws Exception {
            if (source == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "source", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$Mediator", "clonePermissions"));
            }
            if (target == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "target", "org/jetbrains/kotlin/com/intellij/openapi/util/io/FileSystemUtil$Mediator", "clonePermissions"));
            }
            return false;
        }
    }
}

