/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server.integration.jbpm3;

import java.util.ArrayList;
import java.util.List;

import javax.ejb.CreateException;

import org.jboss.bpm.console.server.util.ServiceLocator;
import org.jbpm.command.CancelWorkOnTaskCommand;
import org.jbpm.command.DeleteProcessDefinitionCommand;
import org.jbpm.command.DeployProcessCommand;
import org.jbpm.command.GetProcessDefinitionCommand;
import org.jbpm.command.GetProcessDefinitionsCommand;
import org.jbpm.command.GetProcessInstanceCommand;
import org.jbpm.command.GetProcessInstancesCommand;
import org.jbpm.command.GetTaskInstanceCommand;
import org.jbpm.command.GetTaskListCommand;
import org.jbpm.command.NewProcessInstanceCommand;
import org.jbpm.command.SignalCommand;
import org.jbpm.command.StartWorkOnTaskCommand;
import org.jbpm.command.TaskInstanceEndCommand;
import org.jbpm.ejb.LocalCommandService;
import org.jbpm.ejb.LocalCommandServiceHome;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.taskmgmt.exe.TaskInstance;

/**
 * Adopts a business interface to the command facade.<br>
 * Depends on the CommandServiceBean (<code>java:ejb/CommandServiceBean</code>).
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class JBPM3CommandDelegate
{
   private LocalCommandService facade;
   private static final String DEFAULT_JNDI_NAME = "java:ejb/CommandServiceBean";

   /*
   * Lookup with particluar JNDI name.
   * @param jndiName
   */
   public JBPM3CommandDelegate(String jndiName)
   {
      ejbLookup(jndiName);
   }

   /*
   * Lookup using default JNDI name.
   * @see #DEFAULT_JNDI_NAME
   */

   public JBPM3CommandDelegate()
   {
      ejbLookup(DEFAULT_JNDI_NAME);
   }

   private void ejbLookup(String jndiName)
   {
      try
      {
         LocalCommandServiceHome home = (LocalCommandServiceHome)ServiceLocator.getEjbLocalHome(jndiName);
         this.facade = home.create();
      }
      catch (CreateException e)
      {
         throw new RuntimeException("Failed to create LocalCommandService", e);
      }
   }

   public List<ProcessDefinition> getActualDefinitions()
   {
      List<ProcessDefinition> defs = (List<ProcessDefinition>)facade.execute(new GetProcessDefinitionsCommand(false));
      return defs;
   }

   public ProcessDefinition getActualDefinition(String name)
   {
      ProcessDefinition def = (ProcessDefinition)facade.execute(new GetProcessDefinitionCommand(name));
      return def;
   }

   public ProcessDefinition getActualDefinition(long processId)
   {
      List<ProcessDefinition> defs = getActualDefinitions();

      ProcessDefinition match = null;
      for (ProcessDefinition p0 : defs)
      {
         if (processId == p0.getId())
         {
            match = p0;
            break;
         }
      }

      if (null == match)
         throw new IllegalArgumentException("No process definition with ID " + processId);

      return match;
   }

   public void removeActualDefinition(long processId)
   {
      ProcessDefinition def = getActualDefinition(processId);
      facade.execute(new DeleteProcessDefinitionCommand(processId));
   }

   public org.jbpm.graph.exe.ProcessInstance getActualInstance(long instanceId)
   {
      ProcessInstance instance = (ProcessInstance)facade.execute(new GetProcessInstanceCommand(instanceId));
      return instance;
   }

   public List<ProcessInstance> getActualInstances(long processId)
   {
      ProcessDefinition p0 = getActualDefinition(processId);
      GetProcessInstancesCommand command = new GetProcessInstancesCommand();
      command.setProcessId(processId);

      List<ProcessInstance> instances = (List<ProcessInstance>)facade.execute(command);

      return instances;
   }

   public ProcessInstance startNewInstance(long processId)
   {
      NewProcessInstanceCommand command = new NewProcessInstanceCommand();
      command.setProcessId(processId);
      ProcessInstance instance = (ProcessInstance)facade.execute(command);

      // default signal to behave spec compliant
      instance.signal();
      return instance;
   }

   public ProcessDefinition deploy(byte[] data)
   {
      ProcessDefinition p0 = (ProcessDefinition)facade.execute(new DeployProcessCommand(data));
      return p0;
   }

   public List<TaskInstance> getActualTasksForActor(String actorName)
   {

      List<String> groupNames = getGroupsForActor(actorName);

      String[] actors = new String[groupNames.size() + 1];
      int i = 0;
      for (String s : groupNames)
      {
         actors[i] = s;
         i++;
      }

      actors[i] = actorName; // all groups & the username

      List<TaskInstance> tasks = new ArrayList<TaskInstance>();
      GetTaskListCommand command = new GetTaskListCommand(actors);
      tasks.addAll((List<TaskInstance>)facade.execute(command));
      return tasks;
   }

   public List<String> getGroupsForActor(String actorName)
   {
      List<String> groupNames = (List<String>)facade.execute(new GetGroupMembershipCommand(actorName));
      return groupNames;
   }

   public List<String> getActorsForGroup(String groupName)
   {
      List<String> actorIds = (List<String>)facade.execute(new GetUsersForGroupCommand(groupName));
      return actorIds;
   }

   public TaskInstance getTaskById(long taskId)
   {
      // include variables, but no logs
      return (TaskInstance)facade.execute(new GetTaskInstanceCommand(taskId, true, false));
   }

   public void assignTask(long taskId, String actorId)
   {
      if (actorId != null)
      {
         StartWorkOnTaskCommand command = new StartWorkOnTaskCommand(taskId, true);
         command.setActorId(actorId);
         facade.execute(command);
      }
      else
      {
         CancelWorkOnTaskCommand command = new CancelWorkOnTaskCommand(taskId);
         facade.execute(command);
      }
   }

   public void signalToken(long tokenId, String signalName)
   {
      SignalCommand command = new SignalCommand(tokenId, signalName);
      facade.execute(command);
   }

   public void endTask(long taskId, String signalName)
   {
      TaskInstanceEndCommand command = new TaskInstanceEndCommand();
      command.setTaskInstanceId(taskId);
      command.setTransitionName(signalName);

      facade.execute(command);
   }
}
