/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.webbeans.integration.microcontainer.deployer.env;

import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.ValueMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractSimpleRealDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.webbeans.context.api.BeanStore;
import org.jboss.webbeans.context.api.helpers.ConcurrentHashMapBeanStore;

/**
 * Deploy WebBeans boostrap service.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class WebBeansBootstrapDeployer extends AbstractSimpleRealDeployer<WebBeanDiscoveryEnvironment>
{
   public WebBeansBootstrapDeployer()
   {
      super(WebBeanDiscoveryEnvironment.class);
      setTopLevelOnly(true);
      setStage(DeploymentStages.PRE_REAL);
      addOutput(BeanMetaData.class);
   }

   public void deploy(DeploymentUnit unit, WebBeanDiscoveryEnvironment env) throws DeploymentException
   {
      String unitName = unit.getName();

      String envName = unitName + "_JBossWebBeanDiscovery";
      BeanMetaDataBuilder envWrapper = BeanMetaDataBuilder.createBuilder(envName, "org.jboss.webbeans.integration.jbossas.bootstrap.JBossWebBeanDiscovery");
      envWrapper.addConstructorParameter(WebBeanDiscoveryEnvironment.class.getName(), env);
      unit.addAttachment(envName + "_" + BeanMetaData.class.getSimpleName(), envWrapper.getBeanMetaData());

      String bootstrapName = unitName + "_WebBeansBootstrap";
      BeanMetaDataBuilder bootstrap = BeanMetaDataBuilder.createBuilder(bootstrapName, "org.jboss.webbeans.bootstrap.WebBeansBootstrap");
      bootstrap.addPropertyMetaData("webBeanDiscovery", bootstrap.createInject(envName));
      bootstrap.addPropertyMetaData("ejbDiscovery", createEjbConnector("JBossEjbDiscovery", "org.jboss.webbeans.integration.ejb.JBossEjbDiscovery", unit));
      bootstrap.addPropertyMetaData("ejbResolver", createEjbConnector("JBossEjbResolver", "org.jboss.webbeans.integration.ejb.JBossEjbResolver", unit));
      bootstrap.addPropertyMetaData("transactionServices", bootstrap.createInject("JBossTransactionServices"));
      bootstrap.addPropertyMetaData("applicationContext", createBeanStore());
      bootstrap.setCreate("initialize");
      bootstrap.setStart("boot");
      bootstrap.setDestroy("shutdown");
      bootstrap.addDependency("RealTransactionManager"); // so we know TM is present in JBossTransactionServices 
      unit.addAttachment(bootstrapName + "_" + BeanMetaData.class.getSimpleName(), bootstrap.getBeanMetaData());
   }

   /**
    * Create new bean store.
    *
    * @return the bean store instance
    */
   protected BeanStore createBeanStore()
   {
      return new ConcurrentHashMapBeanStore();
   }

   /**
    * Create ejb connector.
    *
    * @param name the connector name
    * @param bean the bean to create
    * @param unit the deployment unit
    * @return new inject metadata
    */
   protected ValueMetaData createEjbConnector(String name, String bean, DeploymentUnit unit)
   {
      String beanName = unit.getName() + "_" + name;
      BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(beanName, bean);
      builder.setFactory(name);
      builder.setFactoryMethod("createBean");
      builder.addPropertyMetaData("deploymentUnit", unit);
      unit.addAttachment(beanName + "_" + BeanMetaData.class.getSimpleName(), builder.getBeanMetaData());
      return builder.createInject(beanName);
   }
}
