/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.hierarchy.plugins;

import java.net.URL;

import org.jboss.classloading.spi.visitor.ClassFilter;
import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.InterfaceInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.scanning.plugins.helpers.ResourceOwnerFinder;
import org.jboss.scanning.plugins.visitor.ReflectProvider;
import org.jboss.scanning.plugins.visitor.ReflectResourceVisitor;

/**
 * Hierarchy type visitor.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class HierarchyTypeVisitor extends ReflectResourceVisitor
{
   private ResourceOwnerFinder finder;
   private HierarchyIndexImpl index;
   private TypeInfo OBJECT;

   public HierarchyTypeVisitor(ReflectProvider provider, ResourceOwnerFinder finder, HierarchyIndexImpl index)
   {
      super(provider);
      if (finder == null)
         throw new IllegalArgumentException("Null finder");
      if (index == null)
         throw new IllegalArgumentException("Null index");
      this.finder = finder;
      this.index = index;
   }

   public ResourceFilter getFilter()
   {
      return ClassFilter.INSTANCE;
   }

   @Override
   protected void handleClass(ResourceContext resource, ClassInfo classInfo) throws Exception
   {
      URL ownerURL = finder.findOwnerURL(resource);
      String path = HierarchyIndexImpl.fixPath(ownerURL.getPath());
      handleInterfaces(path, classInfo, classInfo); // handle target's interfaces
      recurse(path, classInfo, classInfo.getSuperclass()); // recurse on super class
   }

   private void recurse(String path, final ClassInfo ci, ClassInfo sub)
   {
      if (sub == null)
         return;

      if (OBJECT == null)
         OBJECT = sub.getTypeInfoFactory().getTypeInfo(Object.class);

      if (sub == OBJECT) // no need to index Object sub-types
         return;

      index.putInfo(path, ci, sub); // sub on ref
      handleInterfaces(path, ci, sub); // sub on ref's interfaces

      // recurse
      recurse(path, ci, sub.getSuperclass());
   }

   private void handleInterfaces(String path, ClassInfo ci, ClassInfo ref)
   {
      InterfaceInfo[] ifaces = ref.getInterfaces();
      if (ifaces != null)
      {
         for (InterfaceInfo ii : ifaces)
         {
            recurse(path, ci, ii); // recurse on interface
         }
      }
   }
}