/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.annotations.plugins;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.annotation.Annotation;

import org.jboss.metadata.spi.signature.Signature;
import org.jboss.util.JBossObject;

/**
 * Class name and signature pair.
 * With those two we can re-create annotation value.
 *
 * If the keepAnnotations flag is on in DefaultAnnotationRepository
 * we cache the annotation value from GenericAnnotationResourceVisitor.
 *
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
public class ClassSignaturePair extends JBossObject implements Externalizable
{
   /** The serial version UID */
   private static final long serialVersionUID = 1L;

   private String className;
   private Signature signature;
   private Annotation annotation;

   /**
    * Should only be used for de-serialization.
    */
   public ClassSignaturePair()
   {
   }

   public ClassSignaturePair(String className, Signature signature)
   {
      this(className, signature, null);
   }

   public ClassSignaturePair(String className, Signature signature, Annotation annotation)
   {
      if (className == null)
         throw new IllegalArgumentException("Null class name");

      this.className = className;
      this.signature = signature;
      this.annotation = annotation;
   }

   public void writeExternal(ObjectOutput out) throws IOException
   {
      out.writeUTF(className);
      // signature
      boolean bs = signature != null;
      out.writeBoolean(bs);
      if (bs)
         out.writeObject(new SignatureSerializer(signature));
   }

   public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
   {
      className = in.readUTF();
      if (in.readBoolean())
      {
         SignatureSerializer ss = (SignatureSerializer) in.readObject();
         signature = ss.getSignature();
      }
   }

   /**
    * Get the classname.
    *
    * @return the classname
    */
   public String getClassName()
   {
      return className;
   }

   /**
    * Get the signature.
    *
    * @return the signature
    */
   public Signature getSignature()
   {
      return signature;
   }

   /**
    * Get the annotation.
    *
    * @return the annotation
    */
   public Annotation getAnnotation()
   {
      return annotation;
   }

   protected int getHashCode()
   {
      int hash = className.hashCode();
      if (signature != null)
         hash += 7 * signature.hashCode();
      return hash;
   }

   public boolean equals(Object obj)
   {
      if (obj instanceof ClassSignaturePair == false)
         return false;

      ClassSignaturePair csPair = (ClassSignaturePair)obj;
      if (className.equals(csPair.getClassName()))
         return equals(signature, csPair.getSignature());
      else
         return false;
   }
}
