/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins.helpers;

import java.lang.reflect.Method;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;

import org.jboss.classloading.spi.visitor.ResourceContext;

/**
 * Direct root finder.
 * 
 * We asume underlying resource context implements RootAwareResource.
 * If this is not the case, use different ROF impl.
 *
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
public class DirectRootResourceOwnerFinder implements ResourceOwnerFinder
{
   private static ResourceOwnerFinder instance;
   private static Method currentRootGetter;

   static
   {
      try
      {
         SecurityManager sm = System.getSecurityManager();
         if (sm == null)
            currentRootGetter = getLookupMethod();
         else
            currentRootGetter = AccessController.doPrivileged(new PrivilegedExceptionAction<Method>()
            {
               public Method run() throws Exception
               {
                  return getLookupMethod();
               }
            });
      }
      catch (Throwable ignored)
      {
      }
   }

   private static Method getLookupMethod() throws Exception
   {
      ClassLoader cl = ResourceContext.class.getClassLoader();
      Class<?> clazz = cl.loadClass("org.jboss.classloading.spi.visitor.RootAwareResource");
      return clazz.getDeclaredMethod("getRootUrl");
   }

   private DirectRootResourceOwnerFinder()
   {
   }

   public static boolean isValid()
   {
      return currentRootGetter != null;
   }

   public static ResourceOwnerFinder getInstance()
   {
      if (isValid() == false)
         throw new IllegalArgumentException("Invalid finder, cannot access root url directly.");

      if (instance == null)
         instance = new DirectRootResourceOwnerFinder();
      return instance;
   }

   /**
    * This one loads the class.
    *
    * @param resource the resource
    * @return the owner url
    */
   public URL findOwnerURL(ResourceContext resource)
   {
      try
      {
         return (URL) currentRootGetter.invoke(resource);
      }
      catch (RuntimeException e)
      {
         throw e;
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }
}