/*
 * JBoss, Home of Professional Open Source
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.embedded.api.server;

import org.jboss.bootstrap.api.as.config.JBossASServerConfig;
import org.jboss.bootstrap.api.as.server.JBossASServerFactory;

/**
 * JBossASEmbeddedServerFactory
 * 
 * Stateless factory used in constructing new {@link JBossASEmbeddedServer}
 * instances.  Does not leak internals and allows the client compilation and 
 * runtime classpath (application ClassLoader) to be free of the 
 * implementation classes.
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class JBossASEmbeddedServerFactory
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * The FQN of the default server implementation class to create
    */
   private static final String DEFAULT_SERVER_IMPL_CLASS_NAME = "org.jboss.embedded.core.server.JBossASEmbeddedServerImpl";

   /**
    * System property denoting the location of the temp directory
    */
   private static final String SYS_PROP_TMP_DIR = "java.io.tmpdir";

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor, should not be used
    * 
    * @throws UnsupportedOperationException If invoked
    */
   private JBossASEmbeddedServerFactory()
   {
      // No access
      throw new UnsupportedOperationException("No instances permitted");
   }

   //-------------------------------------------------------------------------------------||
   // Factory Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a JBossASEmbeddedServer from the default implementation class name
    * {@link JBossASEmbeddedServerFactory#DEFAULT_SERVER_IMPL_CLASS_NAME}, using the 
    * Thread Context ClassLoader.  The returned server will additionally be created using
    * the TCCL.
    * 
    * This factory method will also set up a boot.log within the system temporary 
    * directory unless another location has been manually specified via the presense 
    * of the system property "jboss.boot.server.log.dir".
    * 
    * @return The newly-created Server
    */
   public static JBossASEmbeddedServer createServer()
   {
      return createServer(SecurityActions.getTccl());
   }

   /**
    * Creates a JBossASEmbeddedServer from the default implementation class name
    * {@link JBossASEmbeddedServerFactory#DEFAULT_SERVER_IMPL_CLASS_NAME}, using the
    * specified ClassLoader.  The returned server will additionally be created using
    * the ClassLoader denoted.
    * 
    * This factory method will also set up a boot.log within the system temporary 
    * directory unless another location has been manually specified via the presense 
    * of the system property "jboss.boot.server.log.dir".
    * 
    * @param cl The ClassLoader used to create the new server instance
    * @throws IllegalArgumentException If the ClassLoader is null
    * @return The newly-created Server 
    */
   public static JBossASEmbeddedServer createServer(final ClassLoader cl) throws IllegalArgumentException
   {

      /*
       * Set boot log directory to a temp directory
       */
      final String sysPropBootLogDir = "jboss.boot.server.log.dir";
      final String sysPropLogDir = JBossASServerConfig.PROP_KEY_JBOSSAS_SERVER_LOG_DIR;
      final String manualBootLogDir = SecurityActions.getSystemProperty(sysPropBootLogDir);
      final String manualLogDir = SecurityActions.getSystemProperty(sysPropLogDir);
      // If nothing's been explicitly specified
      if (manualBootLogDir == null && manualLogDir == null)
      {
         // We default it
         final String serverLogDir = SecurityActions.getSystemProperty(SYS_PROP_TMP_DIR);
         SecurityActions.setSystemProperty(sysPropBootLogDir, serverLogDir);
         System.out.println("Boot Log available in: " + serverLogDir);
      }
      // If we've got a manual log dir, use it
      else if (manualLogDir != null)
      {
         SecurityActions.setSystemProperty(sysPropBootLogDir, manualLogDir);
      }

      try
      {
         return JBossASServerFactory.createServer(DEFAULT_SERVER_IMPL_CLASS_NAME, cl, JBossASEmbeddedServer.class);
      }
      catch (final InstantiationException ie)
      {
         // Should really only be encountered if the impl class name above is wrong, so don't expose
         // this as a checked exception to the user, wrap in a runtime exception
         throw new RuntimeException("Server instance could not be created", ie);
      }
   }

}
