/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
  *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.embedded.api.server;

import java.io.File;
import java.net.URL;

import org.jboss.bootstrap.api.as.config.JBossASServerConfig;
import org.jboss.bootstrap.api.as.server.JBossASBasedServer;
import org.jboss.embedded.api.Deployable;
import org.jboss.embedded.api.DeploymentException;
import org.jboss.shrinkwrap.api.Archive;

/**
 * JBossASEmbeddedServer
 *
 * Contract of the JBoss Application Server as seen by clients 
 * wishing to embed into an existing runtime
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public interface JBossASEmbeddedServer extends JBossASBasedServer<JBossASEmbeddedServer, JBossASServerConfig>
{
   //-------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Deploys the specified archives into the server as one atomic operation.  Archives
    * already deployed will be ignored. 
    * 
    * @param archives
    * @throws DeploymentException
    * @throws IllegalArgumentException If no archives were specified
    */
   void deploy(Archive<?>... archives) throws DeploymentException, IllegalArgumentException;

   /**
    * Deploys the specified {@link URL}s into the server as one atomic operation.  URLs
    * already deployed will be ignored. 
    * 
    * @param urls
    * @throws DeploymentException
    * @throws IllegalArgumentException If no urls were specified 
    */
   void deploy(URL... urls) throws DeploymentException, IllegalArgumentException;

   /**
    * Deploys the specified {@link File}s into the server as one atomic operation.  Files
    * already deployed will be ignored. 
    * 
    * @param files
    * @throws DeploymentException
    * @throws IllegalArgumentException If no files were specified 
    */
   void deploy(File... files) throws DeploymentException, IllegalArgumentException;

   /**
    * Deploys the specified {@link Deployable}s into the server as one atomic operation.  Deployables
    * already deployed will be ignored. 
    * 
    * @param deployables
    * @throws DeploymentException
    * @throws IllegalArgumentException If no deployables were specified
    */
   void deploy(Deployable... deployables) throws DeploymentException, IllegalArgumentException;

   /**
    * Undeploys the specified archives from the server as one atomic operation.
    * The archives must have been previously deployed, else they will be ignored.
    * 
    * @param archives
    * @throws DeploymentException
    * @throws IllegalArgumentException If no archives were specified
    */
   void undeploy(Archive<?>... archives) throws DeploymentException, IllegalArgumentException;

   /**
    * Undeploys the specified {@link URL}s from the server as one atomic operation. 
    * The URLs must have been previously deployed, else they will be ignored.
    * 
    * @param urls
    * @throws DeploymentException
    * @throws IllegalArgumentException If no urls were specified 
    */
   void undeploy(URL... urls) throws DeploymentException, IllegalArgumentException;

   /**
    * Undeploys the specified {@link File}s from the server as one atomic operation.
    * The files must have been previously deployed, else they will be ignored.
    * 
    * @param files
    * @throws DeploymentException
    * @throws IllegalArgumentException If no files were specified 
    */
   void undeploy(File... files) throws DeploymentException, IllegalArgumentException;

   /**
    * Undeploys the specified {@link Deployable}s from the server as one atomic operation. 
    * The deployables must have been previously deployed, else they will be ignored.
    * 
    * @param deployables
    * @throws DeploymentException
    * @throws IllegalArgumentException If no deployables were specified
    */
   void undeploy(Deployable... deployables) throws DeploymentException, IllegalArgumentException;

}
