/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.ha.cachemanager;

import java.security.AccessController;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheException;
import org.jboss.cache.CacheStatus;
import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.Node;
import org.jboss.cache.NodeNotExistsException;
import org.jboss.cache.Region;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.interceptors.base.CommandInterceptor;
import org.jboss.logging.Logger;
import org.jboss.util.loading.ContextClassLoaderSwitcher;
import org.jgroups.Address;

/**
 * Wrapper around a cache that 1) ensures the calling thread's TCCL doesn't
 * leak to cache threads via calls to create/start; and 2) logs WARNS about calls
 * to stop/destroy as these should be handled by the CacheManager.
 * 
 * TODO disable calls to stop/destroy once testsuite cleanup code is fixed
 * to not call those methods.
 * 
 * @author Brian Stansberry
 */
class CacheManagerManagedCache<K, V> implements Cache<K, V>
{
   private static final Logger log = Logger.getLogger(CacheManagerManagedCache.class);
   
   private final Cache<K, V> delegate;
   private final ContextClassLoaderSwitcher switcher;
   
   CacheManagerManagedCache(Cache<K, V> delegate)
   {
      assert delegate != null : "delegate is null";
      this.delegate = delegate;
      @SuppressWarnings("unchecked")
      ContextClassLoaderSwitcher unchecked = (ContextClassLoaderSwitcher) AccessController.doPrivileged(ContextClassLoaderSwitcher.INSTANTIATOR);
      this.switcher = unchecked;
   }
   
   /** For unit testing */
   Cache<K, V> getDelegate()
   {
      return delegate;
   }

   /**
    * Switches the TCCL to the delegate's classloader before calling create()
    * on the delegate.
    */
   public void create() throws CacheException
   {
      ContextClassLoaderSwitcher.SwitchContext switchContext = switcher.getSwitchContext();
      try
      {
         switchContext.setClassLoader(delegate.getClass().getClassLoader());
         delegate.create();
      }
      finally
      {
         switchContext.reset();
      }
   }

   /**
    * Switches the TCCL to the delegate's classloader before calling start()
    * on the delegate.
    */
   public void start() throws CacheException
   {
      ContextClassLoaderSwitcher.SwitchContext switchContext = switcher.getSwitchContext();
      try
      {
         switchContext.setClassLoader(delegate.getClass().getClassLoader());
         delegate.start();
      }
      finally
      {
         switchContext.reset();
      }
   }

   /**
    * TODO: Log a WARN and do nothing else; currently logs and then calls 
    * through to delegate.
    */
   public void stop()
   {
      log.warn("stop() should not be directly called on caches obtained from a CacheManager -- use CacheManager.releaseCache()", 
            new UnsupportedOperationException("stop() is not supported"));
      delegate.stop();
   }

   /**
    * TODO: Log a WARN and do nothing else; currently logs and then calls 
    * through to delegate.
    */
   public void destroy()
   {
      log.warn("destroy() should not be directly called on caches obtained from a CacheManager -- use CacheManager.releaseCache()", 
            new UnsupportedOperationException("destroy() is not supported"));
      delegate.destroy();
   }
   
   // ------------------------------------------------------------------ Cache
   
   public void addCacheListener(Object arg0)
   {
      delegate.addCacheListener(arg0);
   }

   public void addInterceptor(CommandInterceptor arg0, Class<? extends CommandInterceptor> arg1)
   {
      delegate.addInterceptor(arg0, arg1);      
   }

   public void addInterceptor(CommandInterceptor arg0, int arg1)
   {
      delegate.addInterceptor(arg0, arg1); 
   }

   public void removeInterceptor(Class<? extends CommandInterceptor> arg0)
   {
      delegate.removeInterceptor(arg0);
   }

   public void removeInterceptor(int arg0)
   {
      delegate.removeInterceptor(arg0);
   }

   public void startBatch()
   {
      delegate.startBatch();      
   }

   public void endBatch(boolean arg0)
   {
      delegate.endBatch(arg0);
   }

   public void clearData(String arg0)
   {
      delegate.clearData(arg0);
   }

   public void clearData(Fqn arg0)
   {
      delegate.clearData(arg0);
   }

   public void evict(Fqn arg0)
   {
      delegate.evict(arg0);
   }

   public void evict(Fqn arg0, boolean arg1)
   {
      delegate.evict(arg0, arg1);
   }

   public V get(Fqn arg0, K arg1)
   {
      return delegate.get(arg0, arg1);
   }

   public V get(String arg0, K arg1)
   {
      return delegate.get(arg0, arg1);
   }

   public Set<Object> getCacheListeners()
   {
      return delegate.getCacheListeners();
   }

   public CacheStatus getCacheStatus()
   {
      return delegate.getCacheStatus();
   }

   public Configuration getConfiguration()
   {
      return delegate.getConfiguration();
   }

   public Map<K, V> getData(Fqn arg0)
   {
      return delegate.getData(arg0);
   }

   public InvocationContext getInvocationContext()
   {
      return delegate.getInvocationContext();
   }

   public Set<K> getKeys(String arg0)
   {
      return delegate.getKeys(arg0);
   }

   public Set<K> getKeys(Fqn arg0)
   {
      return delegate.getKeys(arg0);
   }

   public Address getLocalAddress()
   {
      return delegate.getLocalAddress();
   }

   public List<Address> getMembers()
   {
      return delegate.getMembers();
   }

   public Node<K, V> getNode(Fqn arg0)
   {
      return delegate.getNode(arg0);
   }

   public Node<K, V> getNode(String arg0)
   {
      return delegate.getNode(arg0);
   }

   public Region getRegion(Fqn arg0, boolean arg1)
   {
      return delegate.getRegion(arg0, arg1);
   }

   public Node<K, V> getRoot()
   {
      return delegate.getRoot();
   }

   public String getVersion()
   {
      return delegate.getVersion();
   }

   public void move(Fqn arg0, Fqn arg1) throws NodeNotExistsException
   {
      delegate.move(arg0, arg1);
   }

   public void move(String arg0, String arg1) throws NodeNotExistsException
   {
      delegate.move(arg0, arg1);
   }

   public void put(Fqn arg0, Map<? extends K, ? extends V> arg1)
   {
      delegate.put(arg0, arg1);
   }

   public void put(String arg0, Map<? extends K, ? extends V> arg1)
   {
      delegate.put(arg0, arg1);
   }

   public V put(Fqn arg0, K arg1, V arg2)
   {
      return delegate.put(arg0, arg1, arg2);
   }

   public V put(String arg0, K arg1, V arg2)
   {
      return delegate.put(arg0, arg1, arg2);
   }

   public void putForExternalRead(Fqn arg0, K arg1, V arg2)
   {
      delegate.putForExternalRead(arg0, arg1, arg2);
   }

   public V remove(Fqn arg0, K arg1)
   {
      return delegate.remove(arg0, arg1);
   }

   public V remove(String arg0, K arg1)
   {
      return delegate.remove(arg0, arg1);
   }

   public void removeCacheListener(Object arg0)
   {
      delegate.removeCacheListener(arg0);
   }

   public boolean removeNode(Fqn arg0)
   {
       return delegate.removeNode(arg0);
   }

   public boolean removeNode(String arg0)
   {
      return delegate.removeNode(arg0);
   }

   public boolean removeRegion(Fqn arg0)
   {
      return delegate.removeRegion(arg0);
   }

   public void setInvocationContext(InvocationContext arg0)
   {
      delegate.setInvocationContext(arg0);
   }

   // --------------------------------------------------------------  Overrides

   @Override
   public boolean equals(Object obj)
   {
      if (obj instanceof CacheManagerManagedCache)
      {
         @SuppressWarnings("unchecked")
         CacheManagerManagedCache<K, V> other = (CacheManagerManagedCache<K, V>) obj;
         return delegate.equals(other.delegate);
      }
      return false;
   }

   @Override
   public int hashCode()
   {
      return delegate.hashCode();
   }

   public Set<Object> getChildrenNames(Fqn fqn)
   {
      return delegate.getChildrenNames(fqn);
   }

   public Set<String> getChildrenNames(String fqn)
   {
      return delegate.getChildrenNames(fqn);
   }

   public boolean isLeaf(Fqn fqn)
   {
      return delegate.isLeaf(fqn);
   }

   public boolean isLeaf(String fqn)
   {
      return delegate.isLeaf(fqn);
   }
   
   
}
