/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.as.lifecycle;

import org.jboss.bootstrap.api.lifecycle.LifecycleEventException;
import org.jboss.bootstrap.api.lifecycle.LifecycleEventHandler;
import org.jboss.bootstrap.api.lifecycle.LifecycleState;
import org.jboss.bootstrap.api.mc.server.MCBasedServer;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.plugins.event.AbstractEvent;
import org.jboss.kernel.spi.event.KernelEvent;
import org.jboss.kernel.spi.event.KernelEventManager;
import org.jboss.logging.Logger;

/**
 * AbstractKernelEventLifecycleEventHandler
 * 
 * Base class for firing Kernel events in response to 
 * server lifecycle state changes
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public abstract class AbstractKernelEventLifecycleEventHandler implements LifecycleEventHandler
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(AbstractKernelEventLifecycleEventHandler.class);

   //-------------------------------------------------------------------------------------||
   // Instance Members -------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * The server whose kernel we'll use to fire events
    */
   private MCBasedServer<?, ?> server;

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor
    * 
    * @throws IllegalArgumentException If the server was not specified
    */
   public AbstractKernelEventLifecycleEventHandler(final MCBasedServer<?, ?> server) throws IllegalArgumentException
   {
      // Precondition check
      if (server == null)
      {
         throw new IllegalArgumentException("server is required");
      }

      // Set
      this.server = server;
   }

   //-------------------------------------------------------------------------------------||
   // Required Implementations -----------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /* (non-Javadoc)
    * @see org.jboss.bootstrap.spi.lifecycle.LifecycleEventHandler#handleEvent(org.jboss.bootstrap.spi.lifecycle.LifecycleState)
    */
   public final void handleEvent(final LifecycleState state) throws LifecycleEventException
   {
      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Handling server state change to: " + state);
      }

      // Get the Kernel
      final Kernel kernel = this.server.getKernel();

      // Send a notification
      final KernelEventManager eventMgr = kernel.getEventManager();
      final long currentTime = System.currentTimeMillis();
      final String type = this.getNotificationType();
      final KernelEvent event = new AbstractEvent(eventMgr, type, 0, currentTime, new Long(currentTime));
      eventMgr.fireKernelEvent(event);

      // Log
      log.debug("Fired: " + event);
   }

   //-------------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Returns the notification type to be used for the Kernel event
    */
   protected abstract String getNotificationType();

}
