/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.beanvalidation.util;

import java.util.Hashtable;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameParser;
import javax.naming.NamingException;

import javax.validation.ValidatorFactory;
import org.jboss.logging.Logger;
import org.jboss.util.naming.NonSerializableFactory;

/**
 * Default jndi name binder.  This class binds BV factories into golbal JNDI
 * under a well-known subcontext.
 *
 * @author <a href="mailto:stan.silvert@jboss.org">Stan Silvert</a>
 */
public class DefaultJndiBinder implements JndiBinder
{
   protected static Logger log = Logger.getLogger(DefaultJndiBinder.class);
   
   public static final String BV_FACTORY_JNDI_SUBCONTEXT = "BeanValidatorFactories";
   
   private Hashtable jndiEnvironment;
   private Context bvFactoryContext;
   
   public DefaultJndiBinder()
   {
      this(new Hashtable(1));
   }
   
   public DefaultJndiBinder(Hashtable jndiEnvironment)
   {
      setEnvironment(jndiEnvironment);
      createSubContextForFactories();
   }
   
   protected void createSubContextForFactories()
   {
      try
      {
         Context root = new InitialContext(jndiEnvironment);
         this.bvFactoryContext = root.createSubcontext(BV_FACTORY_JNDI_SUBCONTEXT);
      }
      catch (NamingException e)
      {
         log.error("Unable to create JNDI subcontext for Bean Validation Factories", e);
      }
   }
   
   /**
    * Set JNDI environment making sure to add the NonSerializableFactory as an object factory.
    *
    * @param environment the env
    */
   protected void setEnvironment(Hashtable environment)
   {
      setNonSerializableFactory(environment);
      this.jndiEnvironment = environment;
   }
   
   protected void setNonSerializableFactory(Hashtable<String, String> environment)
   {
      String nonSerializableFactory = NonSerializableFactory.class.getName();
      String objFactory = environment.get(Context.OBJECT_FACTORIES);
      if (objFactory != null)
      {
         objFactory = nonSerializableFactory + ":" + objFactory;
      }
      else
      {
         objFactory = nonSerializableFactory;
      }
      
      environment.put(Context.OBJECT_FACTORIES, objFactory);
   }
   
   public void bind(String name, ValidatorFactory validatorFactory) throws NamingException
   {
      NonSerializableFactory.rebind(this.bvFactoryContext, name, validatorFactory);
   }

   public void unbind(String name) throws NamingException
   {
      this.bvFactoryContext.unbind(name);
      NonSerializableFactory.unbind(name);
   }
   
}
