/*
  ItsNat Java Web Application Framework
  Copyright (C) 2007-2011 Jose Maria Arranz Santamaria, Spanish citizen

  This software is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 3 of
  the License, or (at your option) any later version.
  This software is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details. You should have received
  a copy of the GNU Lesser General Public License along with this program.
  If not, see <http://www.gnu.org/licenses/>.
*/
package org.itsnat.impl.core.servlet.http;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.LinkedList;
import java.util.Map;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpSession;
import org.itsnat.core.ItsNatException;
import org.itsnat.core.ItsNatServletResponse;
import org.itsnat.impl.core.browser.Browser;
import org.itsnat.impl.core.servlet.DeserialPendingTask;
import org.itsnat.impl.core.servlet.ItsNatServletContextImpl;
import org.itsnat.impl.core.servlet.ItsNatServletImpl;
import org.itsnat.impl.core.servlet.ItsNatServletRequestImpl;
import org.itsnat.impl.core.servlet.ItsNatServletResponseImpl;
import org.itsnat.impl.core.servlet.ItsNatSessionSerializeContainerImpl;
import org.itsnat.impl.core.util.MapListImpl;

/**
 *
 * @author jmarranz
 */
public class ItsNatHttpSessionReplicationCapableImpl extends ItsNatHttpSessionImpl
{
    public static final String SESSION_ATTRIBUTE_NAME = "itsnat_session";
    public static final String SESSION_ATTRIBUTE_NAME_FRAGMENT_COUNT = "itsnat_session_fragment_count";
    public static final String SESSION_ATTRIBUTE_NAME_FRAGMENT_NUM = "itsnat_session_fragment_";

    protected ItsNatHttpSessionReplicationCapableImpl(HttpSession session,ItsNatServletContextImpl context,Browser browser)
    {
        super(session,context,browser);

        this.serialContainer = new ItsNatSessionSerializeContainerImpl(this);  // Por ahora la serializacin slo tiene sentido en este contexto (http session en modo replication capable)

        // Es el caso del comienzo del primer request, el atributo de estado probablemente
        // no haya sido definido (caso de un nodo).
        // Yo creo que no es necesario pues se hace tambin al final del request
        // De todas maneras no hay problema, la aplicacin no se ha usado por parte
        // del usuario y estar vaca la sesin
        writeItsNatHttpSessionToAttribute();
    }


    protected void writeItsNatHttpSessionToAttribute()
    {
        // http://blog.stringbuffer.com/2009/04/http-sessions-and-google-app-enginej.html
        try
        {
            synchronized(session) // Si requiere serializacin no est mal sincronizar
            {
                if (context.isSessionExplicitSerialize())
                {
                    long size = context.getSessionExplicitSerializeFragmentSize(); // Se supone que no cambia, devuelve siempre el mismo valor

                    // Al deserializar se obtiene siempre una instancia nueva
                    byte[] stream = serializeSession(serialContainer);

                    if (size == ItsNatServletContextImpl.SESSION_EXPLICIT_SERIALIZE_ONE_FRAGMENT)
                    {
                        session.setAttribute(SESSION_ATTRIBUTE_NAME,stream);
                    }
                    else
                    {

                        // Dividimos en trozos de tamao size (> 0)
                        // size es long por si acaso pero por ahora el mximo valor que
                        // serializamos es del tamao int de acuerdo con el tipo de datos del length
                        // del array
                        int sizeInt = (int)size;
                        int numFrag = stream.length / sizeInt; // Recuerda que es divisin entera
                        int modulo = (stream.length % sizeInt);
                        if (modulo != 0) numFrag++; // Uno ms
                        // Llegados aqu numFrag nunca ser cero porque nunca el array estar vaco (length == 0)

                        // Antes limpiamos los atributos actuales excedentes para que no queden trozos perdidos
                        Integer numFragObjOld = (Integer)session.getAttribute(SESSION_ATTRIBUTE_NAME_FRAGMENT_COUNT);
                        if (numFragObjOld != null)
                        {
                            // Eliminamos slo los excedentes pues con los dems siempre puede GAE
                            // evitar su transmisin si detecta que el contenido es el mismo
                            int numFragOld = numFragObjOld.intValue();
                            for(int i = numFrag; i < numFragOld; i++)
                                session.removeAttribute(SESSION_ATTRIBUTE_NAME_FRAGMENT_NUM + i);
                        }

                        session.setAttribute(SESSION_ATTRIBUTE_NAME_FRAGMENT_COUNT,new Integer(numFrag));
                        for(int i = 0; i < numFrag; i++)
                        {
                            int sizeFrag = (i != numFrag - 1) ? sizeInt : modulo;
                            byte[] fragment = new byte[sizeFrag];
                            System.arraycopy(stream, i * sizeInt, fragment, 0, sizeFrag);
                            session.setAttribute(SESSION_ATTRIBUTE_NAME_FRAGMENT_NUM + i,fragment);
                        }
                    }
                }
                else  // No serializacin explcita
                {
                    session.setAttribute(SESSION_ATTRIBUTE_NAME,serialContainer);
                }
            }
        }
        catch(Exception ex)
        {
            // Evitamos guardar en la sesin (si el error ocurri al serializar)
            ItsNatSessionSerializeContainerImpl.showError(ex,true);
        }
    }

    public void endOfRequestBeforeSendCode()
    {
        // No se usa, antes se usaba (errneamente) en la simulacin de serializacin
        // eliminar en el futuro, lo dejamos por si acaso
    }

    public void endOfRequest()
    {
        // Es el caso por ejemplo de GAE, se supone que GAE "serializa" las
        // requests incluso entre JVMs

        // De esta manera notificamos a la sesin nativa que serialice
        // de nuevo la sesin, por ejemplo en GAE, pues GAE slo serializa
        // cuando se actualiza via setAttribute de acuerdo con un uso interno de HttpSessionBindingListener
        // http://groups.google.com/group/google-appengine/browse_thread/thread/94b6e2b38ddfe59

        writeItsNatHttpSessionToAttribute();
    }

    protected static ItsNatSessionSerializeContainerImpl readItsNatSessionSerializeContainerFromSessionAttribute(HttpSession session,ItsNatServletContextImpl context)
    {
        // Puede devolver null, no se ha guardado todava en el atributo (nueva sesin)

        try
        {
            if (context.isSessionExplicitSerialize())
            {
                long size = context.getSessionExplicitSerializeFragmentSize(); // Se supone que no cambia, devuelve siempre el mismo valor

                byte[] stream;
                if (size == ItsNatServletContextImpl.SESSION_EXPLICIT_SERIALIZE_ONE_FRAGMENT)
                {
                    stream = (byte[])session.getAttribute(SESSION_ATTRIBUTE_NAME);
                    if (stream == null) return null;
                }
                else
                {
                    // size > 0
                    Integer numFragObj = (Integer)session.getAttribute(SESSION_ATTRIBUTE_NAME_FRAGMENT_COUNT);
                    if (numFragObj == null) return null; // Nueva sesin, no se ha guardado
                    int numFrag = numFragObj.intValue();
                    ByteArrayOutputStream byteArray = new ByteArrayOutputStream();
                    for(int i = 0; i < numFrag; i++)
                    {
                        byte[] fragment = (byte[])session.getAttribute(SESSION_ATTRIBUTE_NAME_FRAGMENT_NUM + i);
                        byteArray.write(fragment);
                    }
                    stream = byteArray.toByteArray();
                }

                return deserializeSession(stream);
            }
            else // No serializacin explcita
            {
                return (ItsNatSessionSerializeContainerImpl)session.getAttribute(SESSION_ATTRIBUTE_NAME);
            }
        }
        catch(Exception ex)
        {
            // Este error puede darse por ejemplo a causa de un cambio de configuracin del ItsNatServletContextImpl
            // existiendo sesiones por ejemplo ya serializadas a archivo en el contenedor de servlets (por un rearranque).
            // En ese caso es posible que sea el cast el que falle por ejemplo (por cambio de mtodo de serializacin)
            // As devolviendo null desecharemos esta sesin "silenciosamente" sin provocar un error.
            ItsNatSessionSerializeContainerImpl.showError(ex,false);
            return null;
        }
    }

    protected static ItsNatHttpSessionImpl readItsNatHttpSessionFromAttribute(HttpSession session,ItsNatServletContextImpl itsNatContext,ItsNatHttpServletRequestImpl itsNatRequest)
    {
        try
        {
            ItsNatSessionSerializeContainerImpl serialContainer = readItsNatSessionSerializeContainerFromSessionAttribute(session,itsNatContext);
            if (serialContainer == null) return null; // No se ha guardado todava en el atributo (nueva sesin)

            ItsNatHttpSessionReplicationCapableImpl itsNatSession = (ItsNatHttpSessionReplicationCapableImpl)serialContainer.getItsNatSession();
            // Puede ser nula la sesin, significa que la lectura fue errnea (as soportamos cambios en el cdigo sin eliminar manualmente las sesiones guardadas en GAE)
            if (itsNatSession == null) return null;

            // A partir de aqu itsNatSession NO es nulo

            if (itsNatSession.session == null) // Es una instancia nueva resultado de de-serializar
            {
                // Pasamos por aqu inmediatamente despus de de-serializar

                // Fue obtenido por serializacin desde otro nodo o bien al cargar el servidor
                // antes del request, la sesin nativa no est definida todava porque es transient
                itsNatSession.session = session;

                itsNatSession.executeDeserialPendingTasks(itsNatContext,itsNatRequest);

                itsNatContext.addItsNatSession(itsNatSession);
            }

            return itsNatSession; // En teora devolvemos aqu una sesin bien deserializada y formada
        }
        catch(Exception ex)
        {
            // Hay que excluir cualquier posibilidad de que por cambios de esquema, de registro de templates
            // de lo que sea ocurra una excepcin en el proceso de deserializacin de una sesin guardada
            // probablemente antes de los cambios dados, pues lo fcil es deshechar la sessin y punto
            // sin que GAE o ItsNat crea que ha habido un error (lo que ocurre si dejamos propagar la excepcin)
            ItsNatSessionSerializeContainerImpl.showError(ex,false);
            return null;
        }
    }

    protected void executeDeserialPendingTasks(ItsNatServletContextImpl itsNatContext,ItsNatServletRequestImpl itsNatRequest) throws Exception
    {
        // Es posible que la deserializacin se produjera no ahora sino al cargarse
        // el servidor o al restaurar una sesin que no se ha tocado
        // desde hace tiempo, en ese caso el servlet no estaba
        // iniciado y algunas tareas quedaron pendientes, ahora que est
        // el servlet iniciado podemos hacerlo
        // Ahora bien, este request pertenece a un servlet concreto y puede haber varios
        // servlets, pero tenemos que despertar a todos los servlets AHORA porque de otra forma
        // al terminar este request se intentarn serializar los documentos de otros servlets
        // que ni siquiera estn correctamente de-serializados

        DeserialPendingTask sessionTask = getSessionDeserialPendingTask();
        if (sessionTask != null)
        {
            ItsNatServletImpl itsNatServlet = itsNatRequest.getItsNatServletImpl();
            ItsNatServletResponse itsNatResponse = itsNatRequest.getItsNatServletResponse();

            sessionTask.process(itsNatServlet,itsNatRequest,itsNatResponse);

            setSessionDeserialPendingTask(null); // Para liberar memoria
        }

        if (hasDeserialPendingTasks())
        {
            ItsNatServletImpl itsNatServlet = itsNatRequest.getItsNatServletImpl();
            String servletName = itsNatServlet.getName();

            ItsNatServletResponse itsNatResponse = itsNatRequest.getItsNatServletResponse();

            ServletContext context = itsNatContext.getServletContext();

            MapListImpl<String,DeserialPendingTask> pendingTasks = getDeserialPendingTasks();

            for(Map.Entry<String,LinkedList<DeserialPendingTask>> entry : pendingTasks.getMap().entrySet() )
            {
                String currServletName = entry.getKey();
                LinkedList<DeserialPendingTask> pendingTasksOfServlet = entry.getValue();
                if (pendingTasksOfServlet == null) continue; // Por si acaso pero es raro que sea nulo

                if (servletName.equals(currServletName))
                {
                    // El servlet que est haciendo la request de verdad
                    for(DeserialPendingTask task : pendingTasksOfServlet)
                    {
                        task.process(itsNatServlet,itsNatRequest,itsNatResponse);
                    }
                }
                else
                {
                    ItsNatServletImpl currItsNatServlet = ItsNatServletImpl.getItsNatServletByName(currServletName);

                    ServletRequest servRequest = itsNatRequest.getServletRequest();
                    ServletResponse servResponse = itsNatResponse.getServletResponse();

                    if (currItsNatServlet == null)
                    {
                        // Despertamos al servlet para que se inicie y se registren los templates etc
                        RequestDispatcher servletDisp = context.getNamedDispatcher(currServletName);
                        // No chequeamos si es null, caso de eliminacin de servlet o similar en una nueva versin de la app.
                        // no merece la pena porque la deserializacin ser errnea, dejamos fallar
                        // aunque perdamos la sessin entera
                        Object currItsNatAction = servRequest.getAttribute("itsnat_action");
                        servRequest.setAttribute("itsnat_action", ItsNatHttpServletImpl.ACTION_SERVLET_WEAK_UP);

                        servletDisp.include(servRequest,servResponse); // Aseguramos as que se inicializa
                        // Lo dejamos como estaba
                        servRequest.removeAttribute("itsnat_action");
                        if (currItsNatAction != null)
                            servRequest.setAttribute("itsnat_action",currItsNatAction);

                        // Ahora debera de estar
                        currItsNatServlet = ItsNatServletImpl.getItsNatServletByName(currServletName);
                    }


                    // Porque este servlet es diferente al que recibe la request, no pasamos
                    // los objetos request y response originales pues los de ItsNat estn vinculados
                    // al servlet, tenemos que crear un par "falsos"
                    // el nico caso problemtico son los templates basados en TemplateSource que son los nicos que necesitan estos objetos
                    ItsNatServletRequestImpl currItsNatServReq = currItsNatServlet.createItsNatServletRequest(servRequest,servResponse,this); // Pasando la sesin como parmetro evitamos que se intente cargar de nuevo
                    ItsNatServletResponseImpl currItsNatServResp = currItsNatServReq.getItsNatServletResponseImpl();

                    for(DeserialPendingTask task : pendingTasksOfServlet)
                    {
                        task.process(currItsNatServlet,currItsNatServReq,currItsNatServResp);
                    }
                }
            }

            clearDeserialPendingTasks(); // Para liberar memoria
        }
    }

    
    public static byte[] serializeSession(ItsNatSessionSerializeContainerImpl serialContainer)
    {
        ByteArrayOutputStream ostream = null;
        try
        {
           ostream = new ByteArrayOutputStream();
           ObjectOutputStream p = new ObjectOutputStream(ostream);
           p.writeObject(serialContainer); // Write the tree to the stream.
           p.flush();
           ostream.close();    // close the file.
        }
        catch (Exception ex)
        {
            try
            {
                if (ostream != null) ostream.close();
            }
            catch(IOException ex2)
            {
                ex.printStackTrace();
                throw new ItsNatException(ex2);
            }
            throw new ItsNatException(ex);
        }

        return ostream.toByteArray();
    }

    public static ItsNatSessionSerializeContainerImpl deserializeSession(byte[] stream)
    {
        ItsNatSessionSerializeContainerImpl serialContainer;
        ByteArrayInputStream istream = null;
        try
        {
            istream = new ByteArrayInputStream(stream);
            ObjectInputStream q = new ObjectInputStream(istream);
            serialContainer = (ItsNatSessionSerializeContainerImpl)q.readObject();
            istream.close();
        }
        catch (Exception ex)
        {
            try
            {
                if (istream != null) istream.close();
            }
            catch(IOException ex2)
            {
                ex.printStackTrace();
                throw new ItsNatException(ex2);
            }
            throw new ItsNatException(ex);
        }

        return serialContainer;
    }
}
