/*
 * Decompiled with CFR 0.152.
 */
package org.jivesoftware.database;

import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.MissingResourceException;
import org.jivesoftware.database.ConnectionProvider;
import org.jivesoftware.database.DefaultConnectionProvider;
import org.jivesoftware.database.EmbeddedConnectionProvider;
import org.jivesoftware.database.ProfiledConnection;
import org.jivesoftware.database.SchemaManager;
import org.jivesoftware.util.ClassUtils;
import org.jivesoftware.util.JiveGlobals;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DbConnectionManager {
    private static final Logger Log = LoggerFactory.getLogger(DbConnectionManager.class);
    private static ConnectionProvider connectionProvider;
    private static final Object providerLock;
    private static boolean profilingEnabled;
    private static boolean transactionsSupported;
    private static boolean streamTextRequired;
    private static boolean maxRowsSupported;
    private static boolean fetchSizeSupported;
    private static boolean subqueriesSupported;
    private static boolean scrollResultsSupported;
    private static boolean batchUpdatesSupported;
    static boolean pstmt_fetchSizeSupported;
    private static final String SETTING_DATABASE_MAX_RETRIES = "database.maxRetries";
    private static final String SETTING_DATABASE_RETRY_DELAY = "database.retryDelay";
    private static DatabaseType databaseType;
    private static SchemaManager schemaManager;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void ensureConnectionProvider() {
        if (connectionProvider != null) {
            return;
        }
        Object object = providerLock;
        synchronized (object) {
            if (connectionProvider != null) {
                return;
            }
            String className = JiveGlobals.getXMLProperty("connectionProvider.className");
            if (className != null) {
                try {
                    Class conClass = ClassUtils.forName(className);
                    DbConnectionManager.setConnectionProvider((ConnectionProvider)conClass.newInstance());
                }
                catch (Exception e) {
                    Log.warn("Failed to create the connection provider specified by connectionProvider.className. Using the default pool.", (Throwable)e);
                    DbConnectionManager.setConnectionProvider(new DefaultConnectionProvider());
                }
            } else {
                DbConnectionManager.setConnectionProvider(new DefaultConnectionProvider());
            }
        }
    }

    public static Connection getConnection() throws SQLException {
        DbConnectionManager.ensureConnectionProvider();
        Integer currentRetryCount = 0;
        Integer maxRetries = JiveGlobals.getXMLProperty(SETTING_DATABASE_MAX_RETRIES, 10);
        Integer retryWait = JiveGlobals.getXMLProperty(SETTING_DATABASE_RETRY_DELAY, 250);
        SQLException lastException = null;
        do {
            try {
                Connection con = connectionProvider.getConnection();
                if (con != null) {
                    if (!profilingEnabled) {
                        return con;
                    }
                    return new ProfiledConnection(con);
                }
            }
            catch (SQLException e) {
                lastException = e;
                Log.info("Unable to get a connection from the database pool (attempt " + currentRetryCount + " out of " + maxRetries + ").", (Throwable)e);
            }
            try {
                Thread.sleep(retryWait.intValue());
            }
            catch (Exception exception) {
                // empty catch block
            }
            Integer n = currentRetryCount;
            Integer n2 = currentRetryCount = Integer.valueOf(currentRetryCount + 1);
        } while (currentRetryCount <= maxRetries);
        throw new SQLException("ConnectionManager.getConnection() failed to obtain a connection after " + currentRetryCount + " retries. The exception from the last attempt is as follows: " + lastException);
    }

    public static Connection getTransactionConnection() throws SQLException {
        Connection con = DbConnectionManager.getConnection();
        if (DbConnectionManager.isTransactionsSupported()) {
            con.setAutoCommit(false);
        }
        return con;
    }

    public static void closeTransactionConnection(PreparedStatement pstmt, Connection con, boolean abortTransaction) {
        DbConnectionManager.closeStatement(pstmt);
        DbConnectionManager.closeTransactionConnection(con, abortTransaction);
    }

    public static void closeTransactionConnection(Connection con, boolean abortTransaction) {
        if (DbConnectionManager.isTransactionsSupported()) {
            try {
                if (abortTransaction) {
                    con.rollback();
                } else {
                    con.commit();
                }
            }
            catch (Exception e) {
                Log.error(e.getMessage(), (Throwable)e);
            }
            try {
                con.setAutoCommit(true);
            }
            catch (Exception e) {
                Log.error(e.getMessage(), (Throwable)e);
            }
        }
        DbConnectionManager.closeConnection(con);
    }

    public static void closeResultSet(ResultSet rs) {
        if (rs != null) {
            try {
                rs.close();
            }
            catch (SQLException e) {
                Log.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    public static void closeStatement(Statement stmt) {
        if (stmt != null) {
            try {
                stmt.close();
            }
            catch (Exception e) {
                Log.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    public static void closeStatement(ResultSet rs, Statement stmt) {
        DbConnectionManager.closeResultSet(rs);
        DbConnectionManager.closeStatement(stmt);
    }

    public static void fastcloseStmt(PreparedStatement pstmt) throws SQLException {
        pstmt.close();
    }

    public static void fastcloseStmt(ResultSet rs, PreparedStatement pstmt) throws SQLException {
        rs.close();
        pstmt.close();
    }

    public static void closeConnection(ResultSet rs, Statement stmt, Connection con) {
        DbConnectionManager.closeResultSet(rs);
        DbConnectionManager.closeStatement(stmt);
        DbConnectionManager.closeConnection(con);
    }

    public static void closeConnection(Statement stmt, Connection con) {
        DbConnectionManager.closeStatement(stmt);
        DbConnectionManager.closeConnection(con);
    }

    public static void closeConnection(Connection con) {
        if (con != null) {
            try {
                con.close();
            }
            catch (Exception e) {
                Log.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    public static PreparedStatement createScrollablePreparedStatement(Connection con, String sql) throws SQLException {
        if (DbConnectionManager.isScrollResultsSupported()) {
            return con.prepareStatement(sql, 1004, 1007);
        }
        return con.prepareStatement(sql);
    }

    public static void scrollResultSet(ResultSet rs, int rowNumber) throws SQLException {
        if (DbConnectionManager.isScrollResultsSupported()) {
            if (rowNumber > 0) {
                try {
                    rs.setFetchDirection(1000);
                    rs.relative(rowNumber);
                }
                catch (SQLException e) {
                    Log.error("Error in JDBC method rs.relative(rowNumber).", (Throwable)e);
                    for (int i = 0; i < rowNumber; ++i) {
                        rs.next();
                    }
                }
            }
        } else {
            for (int i = 0; i < rowNumber; ++i) {
                rs.next();
            }
        }
    }

    public static void limitRowsAndFetchSize(PreparedStatement pstmt, int startIndex, int numResults) {
        int MAX_FETCHRESULTS = 500;
        int maxRows = startIndex + numResults;
        DbConnectionManager.setMaxRows(pstmt, maxRows);
        if (pstmt_fetchSizeSupported) {
            if (scrollResultsSupported) {
                DbConnectionManager.setFetchSize(pstmt, Math.min(500, numResults));
            } else {
                DbConnectionManager.setFetchSize(pstmt, Math.min(500, maxRows));
            }
        }
    }

    public static void setFetchSize(PreparedStatement pstmt, int fetchSize) {
        if (pstmt_fetchSizeSupported) {
            try {
                pstmt.setFetchSize(fetchSize);
            }
            catch (Throwable t) {
                Log.error("Disabling JDBC method pstmt.setFetchSize(fetchSize).", t);
                pstmt_fetchSizeSupported = false;
            }
        }
    }

    public static ConnectionProvider getConnectionProvider() {
        return connectionProvider;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setConnectionProvider(ConnectionProvider provider) {
        Object object = providerLock;
        synchronized (object) {
            if (connectionProvider != null) {
                connectionProvider.destroy();
                connectionProvider = null;
            }
            connectionProvider = provider;
            connectionProvider.start();
            Connection con = null;
            try {
                con = connectionProvider.getConnection();
                DbConnectionManager.setMetaData(con);
                schemaManager.checkOpenfireSchema(con);
            }
            catch (MissingResourceException mre) {
                Log.error(mre.getMessage());
            }
            catch (Exception e) {
                Log.error(e.getMessage(), (Throwable)e);
            }
            finally {
                DbConnectionManager.closeConnection(con);
            }
        }
        JiveGlobals.setXMLProperty("connectionProvider.className", provider.getClass().getName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void destroyConnectionProvider() {
        Object object = providerLock;
        synchronized (object) {
            if (connectionProvider != null) {
                connectionProvider.destroy();
                connectionProvider = null;
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static String getLargeTextField(ResultSet rs, int columnIndex) throws SQLException {
        if (!DbConnectionManager.isStreamTextRequired()) return rs.getString(columnIndex);
        try (Reader bodyReader = rs.getCharacterStream(columnIndex);){
            int len;
            if (bodyReader == null) {
                String string = null;
                return string;
            }
            char[] buf = new char[256];
            StringWriter out = new StringWriter(256);
            while ((len = bodyReader.read(buf)) >= 0) {
                out.write(buf, 0, len);
            }
            String value = out.toString();
            out.close();
            return value;
        }
        catch (Exception e) {
            Log.error(e.getMessage(), (Throwable)e);
            throw new SQLException("Failed to load text field");
        }
    }

    public static void setLargeTextField(PreparedStatement pstmt, int parameterIndex, String value) throws SQLException {
        if (DbConnectionManager.isStreamTextRequired()) {
            try {
                StringReader bodyReader = new StringReader(value);
                pstmt.setCharacterStream(parameterIndex, (Reader)bodyReader, value.length());
            }
            catch (Exception e) {
                Log.error(e.getMessage(), (Throwable)e);
                throw new SQLException("Failed to set text field.");
            }
        } else {
            pstmt.setString(parameterIndex, value);
        }
    }

    public static void setMaxRows(Statement stmt, int maxRows) {
        if (DbConnectionManager.isMaxRowsSupported()) {
            try {
                stmt.setMaxRows(maxRows);
            }
            catch (Throwable t) {
                Log.error("Disabling JDBC method stmt.setMaxRows(maxRows).", t);
                maxRowsSupported = false;
            }
        }
    }

    public static void setFetchSize(ResultSet rs, int fetchSize) {
        if (DbConnectionManager.isFetchSizeSupported()) {
            try {
                rs.setFetchSize(fetchSize);
            }
            catch (Throwable t) {
                Log.error("Disabling JDBC method rs.setFetchSize(fetchSize).", t);
                fetchSizeSupported = false;
            }
        }
    }

    public static SchemaManager getSchemaManager() {
        return schemaManager;
    }

    private static void setMetaData(Connection con) throws SQLException {
        DatabaseMetaData metaData = con.getMetaData();
        transactionsSupported = metaData.supportsTransactions();
        subqueriesSupported = metaData.supportsCorrelatedSubqueries();
        try {
            scrollResultsSupported = metaData.supportsResultSetType(1004);
        }
        catch (Exception e) {
            scrollResultsSupported = false;
        }
        batchUpdatesSupported = metaData.supportsBatchUpdates();
        streamTextRequired = false;
        maxRowsSupported = true;
        fetchSizeSupported = true;
        String dbName = metaData.getDatabaseProductName().toLowerCase();
        String driverName = metaData.getDriverName().toLowerCase();
        if (dbName.indexOf("oracle") != -1) {
            databaseType = DatabaseType.oracle;
            streamTextRequired = true;
            scrollResultsSupported = false;
            if (driverName.indexOf("auguro") != -1) {
                streamTextRequired = false;
                fetchSizeSupported = true;
                maxRowsSupported = false;
            }
        } else if (dbName.indexOf("postgres") != -1) {
            databaseType = DatabaseType.postgresql;
            scrollResultsSupported = false;
            fetchSizeSupported = false;
        } else if (dbName.indexOf("interbase") != -1) {
            databaseType = DatabaseType.interbase;
            fetchSizeSupported = false;
            maxRowsSupported = false;
        } else if (dbName.indexOf("sql server") != -1) {
            databaseType = DatabaseType.sqlserver;
            if (driverName.indexOf("una") != -1) {
                fetchSizeSupported = true;
                maxRowsSupported = false;
            }
        } else if (dbName.indexOf("mysql") != -1) {
            databaseType = DatabaseType.mysql;
            transactionsSupported = false;
        } else if (dbName.indexOf("hsql") != -1) {
            databaseType = DatabaseType.hsqldb;
        } else if (dbName.indexOf("db2") != 1) {
            databaseType = DatabaseType.db2;
        }
    }

    public static DatabaseType getDatabaseType() {
        return databaseType;
    }

    public static boolean isProfilingEnabled() {
        return profilingEnabled;
    }

    public static void setProfilingEnabled(boolean enable) {
        if (!profilingEnabled && enable) {
            ProfiledConnection.start();
        } else if (profilingEnabled && !enable) {
            ProfiledConnection.stop();
        }
        profilingEnabled = enable;
    }

    public static boolean isTransactionsSupported() {
        return transactionsSupported;
    }

    public static boolean isStreamTextRequired() {
        return streamTextRequired;
    }

    public static boolean isMaxRowsSupported() {
        return maxRowsSupported;
    }

    public static boolean isFetchSizeSupported() {
        return fetchSizeSupported;
    }

    public static boolean isPstmtFetchSizeSupported() {
        return pstmt_fetchSizeSupported;
    }

    public static boolean isSubqueriesSupported() {
        return subqueriesSupported;
    }

    public static boolean isScrollResultsSupported() {
        return scrollResultsSupported;
    }

    public static boolean isBatchUpdatesSupported() {
        return batchUpdatesSupported;
    }

    public static boolean isEmbeddedDB() {
        return connectionProvider != null && connectionProvider instanceof EmbeddedConnectionProvider;
    }

    public static String getTestSQL(String driver) {
        if (driver == null) {
            return "select 1";
        }
        if (driver.contains("db2")) {
            return "select 1 from sysibm.sysdummy1";
        }
        if (driver.contains("oracle")) {
            return "select 1 from dual";
        }
        return "select 1";
    }

    static {
        providerLock = new Object();
        profilingEnabled = false;
        pstmt_fetchSizeSupported = true;
        databaseType = DatabaseType.unknown;
        schemaManager = new SchemaManager();
    }

    public static enum DatabaseType {
        oracle,
        postgresql,
        mysql,
        hsqldb,
        db2,
        sqlserver,
        interbase,
        unknown;

    }
}

