/*
 * Decompiled with CFR 0.152.
 */
package org.hyperledger.fabric.shim.impl;

import java.util.Map;
import java.util.Properties;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.logging.Logger;
import org.hyperledger.fabric.Logging;
import org.hyperledger.fabric.metrics.Metrics;
import org.hyperledger.fabric.protos.peer.ChaincodeID;
import org.hyperledger.fabric.protos.peer.ChaincodeMessage;
import org.hyperledger.fabric.shim.ChaincodeBase;
import org.hyperledger.fabric.shim.impl.ChaincodeInvocationTask;
import org.hyperledger.fabric.shim.impl.ChaincodeMessageFactory;
import org.hyperledger.fabric.shim.impl.InvocationTaskExecutor;

public final class InvocationTaskManager {
    private static final Logger LOGGER = Logger.getLogger(InvocationTaskManager.class.getName());
    private static final Logger PERFLOGGER = Logger.getLogger("org.hyperledger.Performance");
    private static final String CANNOT_HANDLE_FORMAT = "[%-8.8s] Received %s: cannot handle";
    private static final int SHUTDOWN_TIMEOUT = 60;
    private final Map<String, ChaincodeInvocationTask> innvocationTasks = new ConcurrentHashMap<String, ChaincodeInvocationTask>();
    private Consumer<ChaincodeMessage> outgoingMessage;
    private final ChaincodeBase chaincode;
    private final ChaincodeID chaincodeId;
    private final int queueSize;
    private final int maximumPoolSize;
    private final int corePoolSize;
    private final long keepAliveTime;
    private static final TimeUnit UNIT = TimeUnit.MILLISECONDS;
    private final BlockingQueue<Runnable> workQueue;
    private final ThreadFactory threadFactory = new ThreadFactory(){
        private final AtomicInteger next = new AtomicInteger(0);

        @Override
        public Thread newThread(Runnable r) {
            Thread thread = Executors.defaultThreadFactory().newThread(r);
            thread.setName("fabric-txinvoke:" + this.next.incrementAndGet());
            return thread;
        }
    };
    private final RejectedExecutionHandler handler = new ThreadPoolExecutor.AbortPolicy();
    private final InvocationTaskExecutor taskService;

    public static InvocationTaskManager getManager(ChaincodeBase chaincode, ChaincodeID chaincodeId) {
        return new InvocationTaskManager(chaincode, chaincodeId);
    }

    public InvocationTaskManager(ChaincodeBase chaincode, ChaincodeID chaincodeId) {
        if (chaincode == null) {
            throw new IllegalArgumentException("chaincode can't be null");
        }
        if (chaincodeId == null) {
            throw new IllegalArgumentException("chaincodeId can't be null");
        }
        this.chaincode = chaincode;
        this.chaincodeId = chaincodeId;
        Properties props = chaincode.getChaincodeConfig();
        this.queueSize = Integer.parseInt((String)props.getOrDefault((Object)"TP_QUEUE_SIZE", "5000"));
        this.maximumPoolSize = Integer.parseInt((String)props.getOrDefault((Object)"TP_MAX_POOL_SIZE", "5"));
        this.corePoolSize = Integer.parseInt((String)props.getOrDefault((Object)"TP_CORE_POOL_SIZE", "5"));
        this.keepAliveTime = Long.parseLong((String)props.getOrDefault((Object)"TP_KEEP_ALIVE_MS", "5000"));
        LOGGER.info(() -> "Max Pool Size [TP_MAX_POOL_SIZE]" + this.maximumPoolSize);
        LOGGER.info(() -> "Queue Size [TP_CORE_POOL_SIZE]" + this.queueSize);
        LOGGER.info(() -> "Core Pool Size [TP_QUEUE_SIZE]" + this.corePoolSize);
        LOGGER.info(() -> "Keep Alive Time [TP_KEEP_ALIVE_MS]" + this.keepAliveTime);
        this.workQueue = new LinkedBlockingQueue<Runnable>(this.queueSize);
        this.taskService = new InvocationTaskExecutor(this.corePoolSize, this.maximumPoolSize, this.keepAliveTime, UNIT, this.workQueue, this.threadFactory, this.handler);
        Metrics.getProvider().setTaskMetricsCollector(this.taskService);
    }

    public void onChaincodeMessage(ChaincodeMessage chaincodeMessage) {
        if (null == chaincodeMessage) {
            throw new IllegalArgumentException("chaincodeMessage is null");
        }
        LOGGER.fine(() -> String.format("[%-8.8s] %s", chaincodeMessage.getTxid(), ChaincodeBase.toJsonString(chaincodeMessage)));
        try {
            this.processChaincodeMessage(chaincodeMessage);
        }
        catch (RuntimeException e) {
            this.shutdown();
            throw e;
        }
    }

    private void processChaincodeMessage(ChaincodeMessage chaincodeMessage) {
        ChaincodeMessage.Type msgType = chaincodeMessage.getType();
        switch (this.chaincode.getState()) {
            case CREATED: {
                if (msgType == ChaincodeMessage.Type.REGISTERED) {
                    this.chaincode.setState(ChaincodeBase.CCState.ESTABLISHED);
                    LOGGER.fine(() -> String.format("[%-8.8s] Received REGISTERED: moving to established state", chaincodeMessage.getTxid()));
                    break;
                }
                LOGGER.warning(() -> String.format(CANNOT_HANDLE_FORMAT, chaincodeMessage.getTxid(), msgType));
                break;
            }
            case ESTABLISHED: {
                if (msgType == ChaincodeMessage.Type.READY) {
                    this.chaincode.setState(ChaincodeBase.CCState.READY);
                    LOGGER.fine(() -> String.format("[%-8.8s] Received READY: ready for invocations", chaincodeMessage.getTxid()));
                    break;
                }
                LOGGER.warning(() -> String.format(CANNOT_HANDLE_FORMAT, chaincodeMessage.getTxid(), msgType));
                break;
            }
            case READY: {
                this.handleMsg(chaincodeMessage, msgType);
                break;
            }
            default: {
                LOGGER.warning(() -> String.format(CANNOT_HANDLE_FORMAT, chaincodeMessage.getTxid(), msgType));
            }
        }
    }

    private void handleMsg(ChaincodeMessage message, ChaincodeMessage.Type msgType) {
        LOGGER.fine(() -> String.format("[%-8.8s] Received %s", message.getTxid(), msgType.toString()));
        switch (msgType) {
            case RESPONSE: 
            case ERROR: {
                this.sendToTask(message);
                break;
            }
            case INIT: 
            case TRANSACTION: {
                this.newTask(message, msgType);
                break;
            }
            default: {
                LOGGER.warning(() -> String.format(CANNOT_HANDLE_FORMAT, message.getTxid(), message.getType()));
            }
        }
    }

    private void sendToTask(ChaincodeMessage message) {
        try {
            PERFLOGGER.fine(() -> "> sendToTask TX::" + message.getTxid());
            String key = message.getChannelId() + message.getTxid();
            ChaincodeInvocationTask task = this.innvocationTasks.get(key);
            if (task == null) {
                this.sendFailure(message, new InterruptedException("Task map missing entry: " + key));
            } else {
                task.postMessage(message);
                PERFLOGGER.fine(() -> "< sendToTask TX::" + message.getTxid());
            }
        }
        catch (InterruptedException e) {
            this.sendFailure(message, e);
        }
    }

    private void sendFailure(ChaincodeMessage message, InterruptedException e) {
        LOGGER.severe(() -> "Failed to send response to the task task " + message.getTxid() + Logging.formatError(e));
        ChaincodeMessage m = ChaincodeMessageFactory.newErrorEventMessage(message.getChannelId(), message.getTxid(), "Failed to send response to task");
        this.outgoingMessage.accept(m);
    }

    private void newTask(ChaincodeMessage message, ChaincodeMessage.Type type) {
        String txid = message.getTxid();
        ChaincodeInvocationTask task = new ChaincodeInvocationTask(message, type, this.outgoingMessage, this.chaincode);
        PERFLOGGER.fine(() -> "> newTask:created TX::" + txid);
        this.innvocationTasks.put(task.getTxKey(), task);
        try {
            PERFLOGGER.fine(() -> "> newTask:submitting TX::" + txid);
            CompletableFuture<Void> response = CompletableFuture.runAsync(() -> task.call(), this.taskService);
            response.thenRun(() -> {
                this.innvocationTasks.remove(task.getTxKey());
                PERFLOGGER.fine(() -> "< newTask:completed TX::" + txid);
            });
            PERFLOGGER.fine(() -> "< newTask:submitted TX::" + txid);
        }
        catch (RejectedExecutionException e) {
            LOGGER.warning(() -> "Failed to submit task " + txid + Logging.formatError(e));
            ChaincodeMessage m = ChaincodeMessageFactory.newErrorEventMessage(message.getChannelId(), txid, "Failed to submit task for processing");
            this.outgoingMessage.accept(m);
        }
    }

    public void setResponseConsumer(Consumer<ChaincodeMessage> outgoingMessage) {
        this.outgoingMessage = outgoingMessage;
    }

    public void register() {
        if (this.outgoingMessage == null) {
            throw new IllegalArgumentException("outgoingMessage is null");
        }
        LOGGER.info(() -> "Registering new chaincode " + String.valueOf(this.chaincodeId));
        this.chaincode.setState(ChaincodeBase.CCState.CREATED);
        this.outgoingMessage.accept(ChaincodeMessageFactory.newRegisterChaincodeMessage(this.chaincodeId));
    }

    public void shutdown() {
        this.taskService.shutdown();
        try {
            if (!this.taskService.awaitTermination(60L, TimeUnit.SECONDS)) {
                this.taskService.shutdownNow();
                if (!this.taskService.awaitTermination(60L, TimeUnit.SECONDS)) {
                    System.err.println("Pool did not terminate");
                }
            }
        }
        catch (InterruptedException ex) {
            this.taskService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }
}

