/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * Copyright (c) 2011, Red Hat Inc. or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Inc.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.hibernate.engine.jdbc.spi;

import java.sql.PreparedStatement;

import org.hibernate.ScrollMode;

/**
 * Contracting for preparing SQL statements
 *
 * @author Steve Ebersole
 */
public interface StatementPreparer {
	/**
	 * Prepare a statement.
	 *
	 * @param sql The SQL the statement to be prepared
	 *
	 * @return the prepared statement
	 */
	public PreparedStatement prepareStatement(String sql);

	/**
	 * Prepare a statement.
	 *
	 * @param sql The SQL the statement to be prepared
	 * @param isCallable Whether to prepare as a callable statement.
	 *
	 * @return the prepared statement
	 */
	public PreparedStatement prepareStatement(String sql, boolean isCallable);

	/**
	 * Get a prepared statement to use for inserting using JDBC3
	 * {@link java.sql.PreparedStatement#getGeneratedKeys getGeneratedKeys} processing.
	 *
	 * @param sql - the SQL for the statement to be prepared
	 * @param autoGeneratedKeys - a flag indicating whether auto-generated keys should be returned; one of<ul>
	 *     <li>{@link PreparedStatement#RETURN_GENERATED_KEYS}</li>
	 *     <li>{@link PreparedStatement#NO_GENERATED_KEYS}</li>
	 *     </li>
	 *
	 * @return the prepared statement
	 *
	 * @see java.sql.Connection#prepareStatement(String, int)
	 */
	public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys);


	/**
	 * Get a prepared statement to use for inserting using JDBC3
	 * {@link java.sql.PreparedStatement#getGeneratedKeys getGeneratedKeys} processing.
	 *
	 * @param sql - the SQL for the statement to be prepared
	 * @param columnNames The name of the columns to be returned in the generated keys result set.
	 *
	 * @return the prepared statement
	 *
	 * @see java.sql.Connection#prepareStatement(String, String[])
	 */
	public PreparedStatement prepareStatement(String sql, String[] columnNames);

	/**
	 * Get a prepared statement for use in loading / querying.
	 *
	 * @param sql The SQL the statement to be prepared
	 * @param isCallable Whether to prepare as a callable statement.
	 * @param scrollMode (optional) scroll mode to be applied to the resulting result set; may be null to indicate
	 * no scrolling should be applied.
	 *
	 * @return the prepared statement
	 */
	public PreparedStatement prepareQueryStatement(String sql, boolean isCallable, ScrollMode scrollMode);
}
