//$Id: $
package org.hibernate.lucene.store;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.hibernate.HibernateException;
import org.hibernate.cfg.Configuration;
import org.hibernate.lucene.Indexed;
import org.hibernate.util.ReflectHelper;
import org.hibernate.util.StringHelper;

/**
 * Create a Lucene directory provider
 * <p/>
 * Lucene directory providers are configured through properties
 *   - hibernate.lucene.default.* and
 *   - hibernate.lucene.<indexname>.*
 *
 * <indexname> properties have precedence over default
 *
 * The implementation is described by
 * hibernate.lucene.[default|indexname].directory_provider
 *
 * If none is defined the default value is FSDirectory
 *
 * @author Emmanuel Bernard
 * @author Sylvain Vieujot
 */
public class DirectoryProviderFactory {
	public List<DirectoryProvider<?>> providers = new ArrayList<DirectoryProvider<?>>();
	private static String LUCENE_PREFIX = "hibernate.lucene.";
	private static String LUCENE_DEFAULT = LUCENE_PREFIX + "default.";
	private static String DEFAULT_DIRECTORY_PROVIDER = FSDirectoryProvider.class.getName();



	public DirectoryProvider<?> createDirectoryProvider(Class<?> entity, Configuration cfg) {
		//get properties
		Properties indexProps = getDirectoryProperties( cfg, entity );
		
		//set up the directory
		String className = indexProps.getProperty( "directory_provider" );
		if ( StringHelper.isEmpty( className ) ) {
			className = DEFAULT_DIRECTORY_PROVIDER;
		}
		DirectoryProvider<?> provider = null;
		try {
			Class<DirectoryProvider> directoryClass = ReflectHelper.classForName(
					className, DirectoryProviderFactory.class
			);
			provider = directoryClass.newInstance();
		}
		catch (Exception e) {
			throw new HibernateException( "Unable to instanciate directory provider: " + className, e );
		}
		provider.initialize( entity, cfg, indexProps );
		int index = providers.indexOf( provider );
		if ( index != -1 ) {
			//share the same Directory provider for the same underlying store
			return providers.get( index );
		}
		else {
			providers.add( provider );
			return provider;
		}
	}

	private static Properties getDirectoryProperties(Configuration cfg, Class<?> entity) {
		Properties props = cfg.getProperties();
		String indexName = LUCENE_PREFIX + getTypeName( entity );
		Properties indexProps = new Properties();
		Properties indexSpecificProps = new Properties();
		for ( Map.Entry entry : props.entrySet() ) {
			String key = (String) entry.getKey();
			if ( key.startsWith( LUCENE_DEFAULT ) ) {
				indexProps.setProperty( key.substring( LUCENE_DEFAULT.length() ), (String) entry.getValue() );
			}
			else if ( key.startsWith( indexName ) ) {
				indexSpecificProps.setProperty( key.substring( indexName.length() ), (String) entry.getValue() );
			}
		}
		indexProps.putAll( indexSpecificProps );
		return indexProps;
	}

	public static String getTypeName(Class<?> clazz) {
		String name = clazz.getAnnotation(Indexed.class).index();
		return "".equals( name ) ? clazz.getName() : name;
	}
}
