//$Id: LuceneEventListener.java 10537 2006-09-29 19:18:26Z epbernard $
package org.hibernate.lucene.event;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.hibernate.HibernateException;
import org.hibernate.cfg.Configuration;
import org.hibernate.event.Initializable;
import org.hibernate.event.PostDeleteEvent;
import org.hibernate.event.PostDeleteEventListener;
import org.hibernate.event.PostInsertEvent;
import org.hibernate.event.PostInsertEventListener;
import org.hibernate.event.PostUpdateEvent;
import org.hibernate.event.PostUpdateEventListener;
import org.hibernate.lucene.DocumentBuilder;
import org.hibernate.lucene.Environment;
import org.hibernate.lucene.Indexed;
import org.hibernate.lucene.store.DirectoryProvider;
import org.hibernate.lucene.store.DirectoryProviderFactory;
import org.hibernate.mapping.PersistentClass;

/**
 * This listener supports setting a parent directory for all generated index files.
 * It also supports setting the analyzer class to be used.
 *
 * @author Gavin King
 * @author Emmanuel Bernard
 * @author Mattias Arbin
 */
//TODO takes care of synchronization and index concurrent index change
public class LuceneEventListener implements PostDeleteEventListener, PostInsertEventListener,
		PostUpdateEventListener, Initializable {

	private Map<Class, DocumentBuilder<Object>> documentBuilders = new HashMap<Class, DocumentBuilder<Object>>();
	//** keep track of the index modifiers per file since 1 index modifier can be present at a time */
	private Map<DirectoryProvider, Lock> indexLock = new HashMap<DirectoryProvider, Lock>();
	private boolean initialized;

	private static final Log log = LogFactory.getLog( LuceneEventListener.class );

	public void initialize(Configuration cfg) {
		if ( initialized ) return;

		Class analyzerClass;
		String analyzerClassName = cfg.getProperty( Environment.ANALYZER_CLASS );
		if ( analyzerClassName != null ) {
			try {
				analyzerClass = Class.forName( analyzerClassName );
			}
			catch (Exception e) {
				throw new HibernateException(
						"Lucene analyzer class '" + analyzerClassName + "' defined in property '" + Environment.ANALYZER_CLASS + "' could not be found.",
						e
				);
			}
		}
		else {
			analyzerClass = StandardAnalyzer.class;
		}
		// Initialize analyzer
		Analyzer analyzer;
		try {
			analyzer = (Analyzer) analyzerClass.newInstance();
		}
		catch (ClassCastException e) {
			throw new HibernateException(
					"Lucene analyzer does not implement " + Analyzer.class.getName() + ": " + analyzerClassName
			);
		}
		catch (Exception e) {
			throw new HibernateException( "Failed to instantiate lucene analyzer with type " + analyzerClassName );
		}

		Iterator iter = cfg.getClassMappings();
		DirectoryProviderFactory factory = new DirectoryProviderFactory();
		while ( iter.hasNext() ) {
			PersistentClass clazz = (PersistentClass) iter.next();
			Class<?> mappedClass = clazz.getMappedClass();
			if ( mappedClass != null ) {
				if ( mappedClass.isAnnotationPresent( Indexed.class ) ) {
					DirectoryProvider provider = factory.createDirectoryProvider( mappedClass, cfg );
					final DocumentBuilder<Object> documentBuilder = new DocumentBuilder<Object>(
							mappedClass, analyzer, provider
					);
					if ( ! indexLock.containsKey( provider ) ) {
						indexLock.put( provider, new ReentrantLock() );
					}
					documentBuilders.put( mappedClass, documentBuilder );
				}
			}
		}
		initialized = true;
	}

	public void onPostDelete(PostDeleteEvent event) {
		DocumentBuilder builder = documentBuilders.get( event.getEntity().getClass() );
		if ( builder != null ) {
			remove( builder, event.getId() );
		}
	}

	public void onPostInsert(PostInsertEvent event) {
		final Object entity = event.getEntity();
		DocumentBuilder<Object> builder = documentBuilders.get( entity.getClass() );
		if ( builder != null ) {
			add( entity, builder, event.getId() );
		}
	}

	public void onPostUpdate(PostUpdateEvent event) {
		final Object entity = event.getEntity();
		DocumentBuilder<Object> builder = documentBuilders.get( entity.getClass() );
		if ( builder != null ) {
			final Serializable id = event.getId();
			remove( builder, id );
			add( entity, builder, id );
		}
	}

	private void remove(DocumentBuilder<?> builder, Serializable id) {
		Term term = builder.getTerm( id );
		log.debug( "removing: " + term );
		DirectoryProvider directoryProvider = builder.getDirectoryProvider();
		Lock lock = indexLock.get( directoryProvider );
		lock.lock();
		try {

			IndexReader reader = IndexReader.open( directoryProvider.getDirectory() );
			reader.deleteDocuments( term );
			reader.close();
		}
		catch (IOException ioe) {
			throw new HibernateException( ioe );
		}
		finally {
			lock.unlock();
		}
	}

	private void add(final Object entity, final DocumentBuilder<Object> builder, final Serializable id) {
		Document doc = builder.getDocument( entity, id );
		if ( log.isDebugEnabled() ) {
			log.debug( "adding: " + doc );
		}
		DirectoryProvider directoryProvider = builder.getDirectoryProvider();
		Lock lock = indexLock.get( directoryProvider );
		lock.lock();
		try {
			IndexWriter writer = new IndexWriter(
					directoryProvider.getDirectory(), builder.getAnalyzer(), false
			); //have been created at init time
			writer.addDocument( doc );
			writer.close();
		}
		catch (IOException ioe) {
			throw new HibernateException( ioe );
		}
		finally {
			lock.unlock();
		}
	}

}
