//$Id: DocumentBuilder.java 10537 2006-09-29 19:18:26Z epbernard $
package org.hibernate.lucene;

import java.beans.Introspector;
import java.io.Serializable;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.Term;
import org.hibernate.AssertionFailure;
import org.hibernate.HibernateException;
import org.hibernate.cfg.annotations.Version;
import org.hibernate.lucene.store.DirectoryProvider;

//TODO handle attribute (only getters are handled currently)
public class DocumentBuilder<T> {

	static {
		Version.touch(); //touch version
	}

	private final List<Member> keywordGetters = new ArrayList<Member>();
	private final List<String> keywordNames = new ArrayList<String>();
	private final List<Member> unstoredGetters = new ArrayList<Member>();
	private final List<String> unstoredNames = new ArrayList<String>();
	private final List<Member> textGetters = new ArrayList<Member>();
	private final List<String> textNames = new ArrayList<String>();

	//private final Class<T> beanClass;
	private final DirectoryProvider directoryProvider;
	private String idKeywordName;
	private final Analyzer analyzer;
	private Float idBoost;

	public DocumentBuilder(Class<?> clazz, Analyzer analyzer, DirectoryProvider directory) {
		//this.beanClass = clazz;
		this.analyzer = analyzer;
		this.directoryProvider = directory;

		for ( Class currClass = clazz; currClass != null ; currClass = currClass.getSuperclass() ) {
			Method[] methods = currClass.getDeclaredMethods();
			for ( int i = 0; i < methods.length ; i++ ) {
				Method method = methods[i];
				Keyword keywordAnn = method.getAnnotation( Keyword.class );
				if ( keywordAnn != null ) {
					String name = getAttributeName( method, keywordAnn.name() );
					if ( keywordAnn.id() ) {
						idKeywordName = name;
						idBoost = getBoost( method );
					}
					else {
						setAccessible( method );
						keywordGetters.add( method );
						keywordNames.add( name );
					}
				}
				Unstored unstoredAnn = method.getAnnotation( Unstored.class );
				if ( unstoredAnn != null ) {
					setAccessible( method );
					unstoredGetters.add( method );
					unstoredNames.add( getAttributeName( method, unstoredAnn.name() ) );
				}
				Text textAnn = method.getAnnotation( Text.class );
				if ( textAnn != null ) {
					textGetters.add( method );
					textNames.add( getAttributeName( method, textAnn.name() ) );
				}
			}
		}

		if ( idKeywordName == null ) {
			throw new HibernateException( "No id Keyword for: " + clazz.getName() );
		}
	}

	private Float getBoost(AnnotatedElement element) {
		if (element == null) return null;
		Boost boost = element.getAnnotation( Boost.class );
		return boost != null ? Float.valueOf( boost.value() ) : null;
	}

	private Object getValue(Member member, T bean) {
		try {
			if ( member instanceof java.lang.reflect.Field ) {
				return ( (java.lang.reflect.Field) member ).get( bean );
			}
			else if ( member instanceof Method ) {
				return ( (Method) member ).invoke( bean );
			}
			else {
				throw new AssertionFailure( "Unexpected member: " + member.getClass().getName() );
			}
		}
		catch (Exception e) {
			throw new IllegalStateException( "Could not get property value", e );
		}
	}

	public Document getDocument(T instance, Serializable id) {
		Document doc = new Document();
		Float boost = getBoost( instance.getClass() );
		if (boost != null) {
			doc.setBoost( boost.floatValue() );
		}
		{
			Field idField = new Field( idKeywordName, id.toString(), Field.Store.YES, Field.Index.UN_TOKENIZED );
			if (idBoost != null) {
				idField.setBoost( idBoost.floatValue() );
			}
			doc.add( idField );
		}
		for ( int i = 0; i < keywordNames.size() ; i++ ) {
			Member member = keywordGetters.get( i );
			Object value = getValue( member, instance );
			if ( value != null ) {
				Field field = new Field( keywordNames.get( i ), toString( value ), Field.Store.YES, Field.Index.UN_TOKENIZED );
				boostField(field, member);
				doc.add( field );
			}
		}
		for ( int i = 0; i < textNames.size() ; i++ ) {
			Member member = textGetters.get( i );
			Object value = getValue( member, instance );
			if ( value != null ) {
				Field field = new Field( textNames.get( i ), toString( value ), Field.Store.YES, Field.Index.TOKENIZED );
				boostField(field, member);
				doc.add( field );
			}
		}
		for ( int i = 0; i < unstoredNames.size() ; i++ ) {
			Member member = unstoredGetters.get( i );
			Object value = getValue( member, instance );
			if ( value != null ) {
				Field field = new Field( unstoredNames.get( i ), toString( value ), Field.Store.NO, Field.Index.TOKENIZED );
				boostField(field, member);
				doc.add( field );
			}
		}

		return doc;
	}

	private void boostField(Field field, Member member) {
		Float boost = getBoost( (AnnotatedElement) member );
		if (boost != null) field.setBoost( boost.floatValue() );
	}

	private static String toString(Object value) {
		return value.toString();
	}

	public Term getTerm(Serializable id) {
		return new Term( idKeywordName, id.toString() );
	}

	private static String getAttributeName(Method method, String name) {
		if( ! "".equals( name ) ) return name; //explicit field name

		//decapitalize
		String methodName = method.getName();
		//FIXME we probably should exclude methods not starting with "get" nor "is"
		int startIndex = 3;
		if( methodName.startsWith("is") ) {
			startIndex = 2;
		}
		return Introspector.decapitalize( methodName.substring( startIndex ) );
	}

	public DirectoryProvider getDirectoryProvider() {
		return directoryProvider;
	}

	public Analyzer getAnalyzer() {
		return analyzer;
	}

	private static void setAccessible(Member member) {
		if ( !Modifier.isPublic( member.getModifiers() ) ) {
			( (AccessibleObject) member ).setAccessible( true );
		}
	}
}
