package org.granite.tide.data;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpSession;

import org.granite.context.GraniteContext;
import org.granite.gravity.Channel;
import org.granite.gravity.Gravity;
import org.granite.logging.Logger;
import org.granite.messaging.webapp.HttpGraniteContext;

import flex.messaging.messages.AsyncMessage;
import flex.messaging.messages.CommandMessage;

public class DataDispatcher {
    
    private static final Logger log = Logger.getLogger(DataDispatcher.class);

    
    private boolean enabled = false;
    private String topic = null;
    private DataTopicParams paramsProvider = null;
    private Gravity gravity = null;
    private String sessionId = null;
    private String clientId = null;
    private String subscriptionId = null;
    
    
	public DataDispatcher(String topic, Class<? extends DataTopicParams> dataTopicParamsClass) {
		GraniteContext graniteContext = GraniteContext.getCurrentInstance();
		if (graniteContext == null || !(graniteContext instanceof HttpGraniteContext))
			return;
		
		this.topic = topic;
		try {
			paramsProvider = dataTopicParamsClass.newInstance();
		}
		catch (Exception e) {
			log.error("Could not instantiate class " + dataTopicParamsClass, e);
		}
		
		gravity = (Gravity)graniteContext.getApplicationMap().get("org.granite.gravity.Gravity");
		HttpSession session = ((HttpGraniteContext)graniteContext).getSession(false);
		if (gravity == null || session == null)
			return;
		
		sessionId = session.getId();
		
		clientId = (String)session.getAttribute("org.granite.gravity.channel.clientId." + topic);
		if (clientId == null)
			return;
		subscriptionId = (String)session.getAttribute("org.granite.gravity.channel.subscriptionId." + topic);
		if (subscriptionId == null)
			return;
		
		enabled = true;
	}
	
	
	public void observe() {
		// Prepare the selector even if we are not yet subscribed
		DataObserveParams params = null;
		if (paramsProvider != null) {
			// Collect selector parameters from component
			params = new DataObserveParams();
			paramsProvider.observes(params);
		}
		
		// Ensure that the current Gravity consumer listens about this data topic and params
		GraniteContext graniteContext = GraniteContext.getCurrentInstance();
		HttpSession session = ((HttpGraniteContext)graniteContext).getSession(false);
		
		@SuppressWarnings("unchecked")
		List<DataObserveParams> selectors = (List<DataObserveParams>)session.getAttribute("org.granite.tide.dataSelectors." + topic);
		if (selectors == null) {
			selectors = new ArrayList<DataObserveParams>();
			session.setAttribute("org.granite.tide.dataSelectors." + topic, selectors);
		}
		
		String dataSelector = (String)session.getAttribute("org.granite.gravity.selector." + topic);
		if (params != null && !DataObserveParams.containsParams(selectors, params)) {
			StringBuilder sb = new StringBuilder("type = 'DATA'");
			
			if (!params.isEmpty())
				selectors.add(params);
			
			if (!selectors.isEmpty()) {
				sb.append(" AND (");
				boolean first = true;
				for (DataObserveParams selector : selectors) {
					if (first)
						first = false;
					else
						sb.append(" OR ");
					sb.append("(");
					selector.append(sb);
					sb.append(")");
				}
				sb.append(")");
			}
			
			session.setAttribute("org.granite.gravity.selector." + topic, sb.toString());
		}
		else if (dataSelector == null) {
			dataSelector = "type = 'UNINITIALIZED'";
			session.setAttribute("org.granite.tide.selector." + topic, dataSelector);
		}
		
		if (!enabled)
			return;
		
		String clientId = (String)session.getAttribute("org.granite.gravity.channel.clientId." + topic);	    				
		if (clientId != null) {
			String subscriptionId = (String)session.getAttribute("org.granite.gravity.channel.subscriptionId." + topic);
			
			CommandMessage message = new CommandMessage();
			message.setClientId(clientId);
			message.setHeader(AsyncMessage.DESTINATION_CLIENT_ID_HEADER, subscriptionId);
			message.setHeader(AsyncMessage.SUBTOPIC_HEADER, "tideDataTopic");
			message.setDestination(topic);
			message.setOperation(CommandMessage.SUBSCRIBE_OPERATION);
			
			message.setHeader(CommandMessage.SELECTOR_HEADER, dataSelector);
			
			gravity.handleMessage(message, true);
		}
	}
	
	
	public void publish(Set<Object[]> dataUpdates) {
		if (!enabled)
			return;
		
		try {
			AsyncMessage message = new AsyncMessage();
			message.setClientId(clientId);
			message.setHeader(AsyncMessage.SUBTOPIC_HEADER, "tideDataTopic");
			message.setDestination(topic);
			message.setHeader("org.granite.tide.data.sessionId", sessionId);
			message.setHeader("type", "DATA");
			if (paramsProvider != null) {
				DataPublishParams params = new DataPublishParams();
				for (Object[] dataUpdate : dataUpdates)
					paramsProvider.publishes(params, dataUpdate[1]);
				
				params.setHeaders(message);
			}
			message.setBody(dataUpdates.toArray());
			
			Channel channel = gravity.getChannel(clientId);
			
			gravity.publishMessage(channel, message);
		}
		catch (Exception e) {
			log.error(e, "Could not dispatch data update");
		}
	}
}
